/**
 * @file SFMT-jump.c
 *
 * @brief do jump using jump polynomial.
 *
 * @author Mutsuo Saito (Hiroshima University)
 * @author Makoto Matsumoto (The University of Tokyo)
 *
 * Copyright (C) 2012 Mutsuo Saito, Makoto Matsumoto,
 * Hiroshima University and The University of Tokyo.
 * All rights reserved.
 *
 * The 3-clause BSD License is applied to this software, see
 * LICENSE.txt
 */

#include <assert.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "sfmt-config.h"

#include "SFMT-params.h"
#include "SFMT.h"
#include "SFMT-jump.h"
#include "SFMT-common.h"

#include "SFMT-jump-alt.h"
#include "SFMT-jump-params.h"

#if defined(__cplusplus)
extern "C" {
#endif

inline static void next_state(sfmt_t * sfmt);

#if defined(HAVE_SSE2)
/**
 * add internal state of src to dest as F2-vector.
 * @param dest destination state
 * @param src source state
 */
inline static void add(sfmt_t *dest, sfmt_t *src) {
    int dp = dest->idx / 4;
    int sp = src->idx / 4;
    int diff = (sp - dp + SFMT_N) % SFMT_N;
    int p;
    int i;
    for (i = 0; i < SFMT_N - diff; i++) {
	p = i + diff;
	dest->state[i].si
	    = _mm_xor_si128(dest->state[i].si, src->state[p].si);
    }
    for (; i < SFMT_N; i++) {
	p = i + diff - SFMT_N;
	dest->state[i].si
	    = _mm_xor_si128(dest->state[i].si, src->state[p].si);
    }
}
#else
inline static void add(sfmt_t *dest, sfmt_t *src) {
    int dp = dest->idx / 4;
    int sp = src->idx / 4;
    int diff = (sp - dp + SFMT_N) % SFMT_N;
    int p;
    int i;
    for (i = 0; i < SFMT_N - diff; i++) {
	p = i + diff;
	for (int j = 0; j < 4; j++) {
	    dest->state[i].u[j] ^= src->state[p].u[j];
	}
    }
    for (; i < SFMT_N; i++) {
	p = i + diff - SFMT_N;
	for (int j = 0; j < 4; j++) {
	    dest->state[i].u[j] ^= src->state[p].u[j];
	}
    }
}
#endif

/**
 * calculate next state
 * @param sfmt SFMT internal state
 */
inline static void next_state(sfmt_t * sfmt) {
    int idx = (sfmt->idx / 4) % SFMT_N;
    w128_t *r1, *r2;
    w128_t * pstate = sfmt->state;

    r1 = &pstate[(idx + SFMT_N - 2) % SFMT_N];
    r2 = &pstate[(idx + SFMT_N - 1) % SFMT_N];
    do_recursion(&pstate[idx],
		 &pstate[idx],
		 &pstate[(idx + SFMT_POS1) % SFMT_N],
		 r1,
		 r2);
    r1 = r2;
    r2 = &pstate[idx];
    sfmt->idx = sfmt->idx + 4;
}

#if 0
/**
 * jump ahead using jump_string
 * @param sfmt SFMT internal state input and output.
 * @param jump_string string which represents jump polynomial.
 */
void SFMT_jump(sfmt_t * sfmt, const char * jump_string) {
    sfmt_t work;
    int index = sfmt->idx;
    int bits;
    memset(&work, 0, sizeof(sfmt_t));
    sfmt->idx = SFMT_N32;

    for (int i = 0; jump_string[i] != '\0'; i++) {
	bits = jump_string[i];
	assert(isxdigit(bits));
	bits = tolower(bits);
	if (bits >= 'a' && bits <= 'f') {
	    bits = bits - 'a' + 10;
	} else {
	    bits = bits - '0';
	}
	bits = bits & 0x0f;
	for (int j = 0; j < 4; j++) {
	    if ((bits & 1) != 0) {
		add(&work, sfmt);
	    }
	    next_state(sfmt);
	    bits = bits >> 1;
	}
    }
    *sfmt = work;
    sfmt->idx = index;
}
#endif

#if defined(__cplusplus)
}
#endif

/**
 * Added for Bridge++ project.
 */

/**
 * generate characterisic polynomial in GF2X form
 * polynomial data stored static in header file
 */
NTL::GF2X sfmt_characteristic_polynomial()
{
  return NTL::GF2XFromBytes(characteristic, sizeof(characteristic));
}

/**
 * jump ahead using jump polynomial
 * @param sfmt SFMT state
 * @param jump jump polynomial
 */
void sfmt_jump(sfmt_t * sfmt, const long step, const NTL::GF2X& chf)
{
  sfmt_jump_t jump_poly;
  sfmt_calculate_jump_polynomial(jump_poly, step, chf);
  sfmt_jump_by_polynomial(sfmt, jump_poly);
}

/**
 * calculate jump ahead polynomial
 * @param jump jump polynomial
 * @param step number of steps (in units of 64bit integer or double precision numbers) to go ahead
 * @param chf characteristic polynomial in GF2X form
 */
void sfmt_calculate_jump_polynomial(
  sfmt_jump_t& jump, 
  const long step, 
  const NTL::GF2X& chf
)
{
  jump.step = step;
  jump.jump_poly = NTL::PowerXMod(step/2, chf);  // 4 N 32bit integer to go

  return;
}

/**
 * jump ahead using jump polynomial
 * @param sfmt SFMT state
 * @param jump jump polynomial
 */
void sfmt_jump_by_polynomial(sfmt_t * sfmt, const sfmt_jump_t& jump)
{
  using namespace NTL;

  sfmt_t work;
  memset(&work, 0, sizeof(sfmt_t));

  int index = sfmt->idx;
  sfmt->idx = SFMT_N32;

  long ndeg = deg(jump.jump_poly);

  for (long i = 0; i <= ndeg; ++i) {
    if (IsOne(coeff(jump.jump_poly, i))) {
      add(&work, sfmt);
    }
    next_state(sfmt);
  }

  *sfmt = work;
  sfmt->idx = index;

  // adjust for odd number of steps
  if ((jump.step % 2) != 0) {
    sfmt_genrand_uint64(sfmt);
  }

  return;
}
