/*!
        @file    $Id:: action_F_Rational.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef ACTION_F_RATIONAL_INCLUDED
#define ACTION_F_RATIONAL_INCLUDED

#include "action.h"
#include "fopr_Rational.h"
#include "force_F_Rational.h"

#include "bridgeIO.h"
using Bridge::vout;

//! action class for rational HMC.

/*!
    This class is for rational HMC, and instantiate Fopr and Force
    for rational approximated fermion operator inside the class.
    While this is convenient, the performance is not good when the
    smearing of gauge field is performed.
    Thus use of Action_F_Rational_frame, for which necessary Fopr
    and Force objects are instantiated outside, is recommended.
                                        [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.                [14 Nov 2012 Y.Namekawa]
 */


//- parameters class
class Parameters_Action_F_Rational : virtual public Parameters
{
 public:
  Parameters_Action_F_Rational();
};
//- end

class Action_F_Rational : public Action {
 private:
  // rational approximation parameters:
  int         m_Np;             // number of poles in rational approx.
  int         m_n_exp, m_d_exp; // numerator and denominator of the exponent
  double      m_x_min, m_x_max; // valid range of approximate sign function
  int         m_Niter;          // max iteration of shiftsolver
  double      m_Stop_cond;      // stopping condition of shift solver
  std::string m_label;          // label of action

  Fopr             *m_fopr;
  Force            *m_fopr_force;
  Fopr_Rational    *m_fopr_langev;
  Fopr_Rational    *m_fopr_H;
  Force_F_Rational *m_force_rational;

  Field *m_U;
  int   m_status_linkv;

  Field m_psf;
  Field m_force;


 public:
  //! constructor requires pointers to Fopr and Force instances.
  Action_F_Rational(Fopr *fopr, Force *fopr_force) :
    Action()
  {
    m_fopr       = fopr;
    m_fopr_force = fopr_force;
  }

  //! destructor. constructed instances are deconstructed in tydyup().
  ~Action_F_Rational()
  {
    tidyup();
  }

  //! setting parameters and creating class instances.
  void set_parameters(const Parameters& params);
  void set_parameters(int Np, int n_exp, int d_exp,
                      double x_min, double x_max,
                      int Niter, double Stop_cond);

  //! set the label of action.
  void set_label(std::string label)
  {
    m_label = label;
    vout.detailed(m_vl, "  label: %s\n", m_label.c_str());
  }

  //! returns the label of action.
  std::string get_label()
  {
    return m_label;
  }

  //! setting gauge configuration.
  void set_config(Field *U)
  {
    m_U = U;
    m_fopr->set_config(U);
  }

  //! to be called when link variable is updated.
  void notify_linkv()
  {
    m_status_linkv = 0;
  }

  //! Langevin step called at the beginning of HMC.
  double langevin(RandomNumbers *);

  //! calculation of Hamiltonian.
  double calcH();

  //! returns the force for updating conjugate momentum.
  const Field force();

 private:

  //! creating instances. called from set_parameters().
  void setup();

  //! destruct class instances constructed in setup()
  void tidyup();
};
#endif
