/*!
        @file    $Id:: action_G_Plaq.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "action_G_Plaq.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("beta", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Action.G_Plaq", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Action_G_Plaq::Parameters_Action_G_Plaq() { append_entry(*this); }
//- end

//====================================================================
void Action_G_Plaq::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double beta;

  int err = 0;
  err += params.fetch_double("beta", beta);

  if (err) {
    vout.crucial(m_vl, "Action_G_Plaq: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(beta);
}


//====================================================================
void Action_G_Plaq::set_parameters(double beta)
{
  //- print input parameters
  vout.general(m_vl, "Action_G_Plaq:\n");
  vout.general(m_vl, "  beta = %8.4f\n", beta);

  //- range check
  // NB. beta == 0 is allowed.

  //- store values
  m_beta = beta;

  //- post-process
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;
  m_force.reset(NinG, Nvol, Ndim);

  m_status_linkv = 0;
}


//====================================================================
double Action_G_Plaq::langevin(RandomNumbers *rand)
{
  double H_U = calcH(); // calculate action H_U=beta*(1-Plaq)*Lvol*6 (SA)

  m_status_linkv = 0;

  return H_U;
}


//====================================================================
double Action_G_Plaq::calcH()
{
  int Lvol = CommonParameters::Lvol();
  int Ndim = CommonParameters::Ndim();

  int Ndim2  = Ndim * (Ndim - 1) / 2;
  int size_U = Lvol * Ndim2;

  double plaq = m_staple.plaquette(*m_U);                    // calculate plaquette (SA)
  double H_U  = m_beta * (1.0 - plaq) * Lvol * Ndim2;        // action (SA)

  vout.general(m_vl, "H_Gplaq    = %18.8f\n", H_U);          // total action (SA)
  vout.general(m_vl, "H_G/dof    = %18.8f\n", H_U / size_U); // action per dof (SA)

  return H_U;
}


//====================================================================
const Field Action_G_Plaq::force()
{
  // Gauge conf m_U (SA)
  if (m_status_linkv == 0) {
    int Nin  = m_U->nin();
    int Nvol = m_U->nvol();
    int Nex  = m_U->nex();
    int Nc   = CommonParameters::Nc();

    double   betaNc = m_beta / Nc;
    Mat_SU_N ut(Nc); // Nc x Nc complex matrix (SA)

    Field_G force(Nvol, Nex);
    Field_G st(Nvol, 1);

    vout.general(m_vl, "  Action_G_Plaq:  %s\n", m_label.c_str());

    for (int mu = 0; mu < Nex; ++mu) {
      m_staple.staple(st, *m_U, mu);
      // Calculate staple for m_U(all site,\mu) (SA)

      /* -->--
         |     |
         |     |
           U_mu
      */
      for (int site = 0; site < Nvol; ++site) {
        ut = m_U->mat(site, mu) * st.mat_dag(site); // U_\mu * (staple)^\dagger (SA)
        ut.at();                                    // anti-hermitian and traceless (SA)
        ut *= -betaNc;
        // force = -beta*{U_\mu *staple^\dagger}_{traceless & anti-hermitian) (SA)
        force.set_mat(site, mu, ut);
      }
    }

    m_force = (Field)force; // set all force at m_force (SA)
    ++m_status_linkv;       // frag showing that force is calculated

    double Fave, Fmax, Fdev;
    m_force.stat(Fave, Fmax, Fdev);

    //- calculate average, max, deviation of force over (site, mu)
    vout.general(m_vl, "    Fplaq_ave = %12.6f  Fplaq_max = %12.6f  Fplaq_dev = %12.6f\n",
                 Fave, Fmax, Fdev);

    return m_force;
  } else {
    vout.general(m_vl, "  Action_G_Plaq returns previous force.\n");
    return m_force;
  }
}


//====================================================================
//============================================================END=====
