/*!
        @file    $Id:: action_G_Plaq_SF.cpp #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "action_G_Plaq_SF.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("beta", 0.0);

    param.Register_double("ct0", 0.0);
    param.Register_double("ct1", 0.0);
    param.Register_double("ct2", 0.0);

    param.Register_double_vector("phi", std::valarray<double>());
    param.Register_double_vector("phipr", std::valarray<double>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Action.G_Plaq_SF", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Action_G_Plaq_SF::Parameters_Action_G_Plaq_SF() { append_entry(*this); }
//- end

//====================================================================
void Action_G_Plaq_SF::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double                beta, ct0, ct1, ct2;
  std::valarray<double> phi, phipr;

  int err = 0;
  err += params.fetch_double("beta", beta);
  err += params.fetch_double("ct0", ct0);
  err += params.fetch_double("ct1", ct1);
  err += params.fetch_double("ct2", ct2);
  err += params.fetch_double_vector("phi", phi);
  err += params.fetch_double_vector("phipr", phipr);

  if (err) {
    vout.crucial(m_vl, "Action_G_Plaq_SF: fetch error, input parameter not found.\n");
    abort();
  }


  double gg = 6.0 / beta;
  double ct = ct0 + ct1 * gg + ct2 * gg * gg;

  set_parameters(beta, &phi[0], &phipr[0], ct);
}


//====================================================================

/*!
  Set parameters for the Wilson plaquette action with the SF boundary.
  <ul>
  <li>m_beta
  <li>m_phi, m_phipr: boundary spatial link
  <li>m_ct: improvement factor for the boundary temporal plaquette.
  </ul>
*/
void Action_G_Plaq_SF::set_parameters(double beta, double *phi, double *phipr, double ct)
{
  //- print input parameters
  vout.general(m_vl, "Action_G_Plaq_SF:\n");
  vout.general(m_vl, "  beta  = %12.6f\n", beta);
  vout.general(m_vl, "  phi1  = %12.6f\n", phi[0]);
  vout.general(m_vl, "  phi2  = %12.6f\n", phi[1]);
  vout.general(m_vl, "  phi3  = %12.6f\n", phi[2]);
  vout.general(m_vl, "  phipr1= %12.6f\n", phipr[0]);
  vout.general(m_vl, "  phipr2= %12.6f\n", phipr[1]);
  vout.general(m_vl, "  phipr3= %12.6f\n", phipr[2]);
  vout.general(m_vl, "  ct    = %12.6f\n", ct);

  //- range check
  // NB. beta,phi,phipr,ct = 0 is allowed.

  //- store values
  m_beta = beta;

  m_ct = ct;

  m_phi   = phi;
  m_phipr = phipr;

  //- post-process
  m_staple.set_parameters(m_phi, m_phipr);

  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  m_force.reset(NinG, Nvol, Ndim);

  m_status_linkv = 0;
}


//====================================================================
double Action_G_Plaq_SF::langevin(RandomNumbers *rand)
{
  double H_U = calcH();

  m_status_linkv = 0;

  return H_U;
}


//====================================================================

/*!
  <ul>
  <li>The Wilson's plaquette gauge action:
\f[
 S_G=-\frac{\beta}{N_c}\sum_p{\rm Re}{\rm Tr} U_p
\f]
  <li>Calculate plaquette using Staples_SF.plaquette() function.
  <li>plaquette is NOT normalized to unity in Staples_SF since we have SF boundary and non-active spatial plaquette at the boundary.
  <li>The boundary improvement factor ct is implemented.
  <ul>
  <li>It is avairable by using plaquette_ct() for the plaquette sum.
  </ul>
  </ul>
 */
double Action_G_Plaq_SF::calcH()
{
  int Nc = CommonParameters::Nc();

  //  double plaq = m_staple.plaquette(m_U);
  double plaq = m_staple.plaquette_ct(*m_U, m_ct);

  //  double H_U = m_beta * (1.0-plaq) *  Lvol * Ndim2;
  double H_U = -1.0 / Nc * m_beta * plaq;

  vout.general(m_vl, "H_Gplaq    = %18.8f\n", H_U);

  return H_U;
}


//====================================================================

/*!
  The force for the Wilson plaquette action with the SF boundary.
  <ul>
  <li>The boundary improvement factor ct is implemented.
  <ul>
  <li>It is avairable by using m_staple.staple_ct for the staple.
  </ul>
  <li>Force for the boundary spatial link is set to zero.
  </ul>
*/
const Field Action_G_Plaq_SF::force()
{
  if (m_status_linkv == 0) {
    int Nin  = m_U->nin();
    int Nvol = m_U->nvol();
    int Nex  = m_U->nex();
    int Nc   = CommonParameters::Nc();

    double   betaNc = m_beta / Nc;
    Mat_SU_N ut(Nc);

    Field_G force(Nvol, Nex);
    Field_G st(Nvol, 1);

    vout.general(m_vl, "  Action_G_Plaq_SF:  %s\n", m_label.c_str());

    for (int mu = 0; mu < Nex; ++mu) {
      //     m_staple.staple(st,m_U,mu);
      m_staple.staple_ct(st, *m_U, mu, m_ct);
      for (int site = 0; site < Nvol; ++site) {
        ut = m_U->mat(site, mu) * st.mat_dag(site);
        ut.at();
        ut *= -betaNc;
        force.set_mat(site, mu, ut);
      }
    }

    m_force = (Field)force;
    ++m_status_linkv;

    double Fave, Fmax, Fdev;
    m_force.stat(Fave, Fmax, Fdev);
    vout.general(m_vl, "    Fave = %12.6f  Fmax = %10.6f  Fdev = %12.6f\n",
                 Fave, Fmax, Fdev);

    return m_force;
  } else {
    vout.general(m_vl, "  Action_G_Plaq_SF returns previous force.\n");
    return m_force;
  }
}


/*!
  <ul>
  <li>The result is (x,y,z,t,mu,color,real part,imaginary part)
  </ul>
 */

//====================================================================
void Action_G_Plaq_SF::print_force(const Field_G *U)
{
  int Nc = CommonParameters::Nc();
  int Nx = CommonParameters::Nx();
  int Ny = CommonParameters::Ny();
  int Nz = CommonParameters::Nz();
  int Nt = CommonParameters::Nt();

  Index_lex idx;
  int       ix;
  Mat_SU_N  Tmp(Nc);

  //  if(comm->ipe(3)==0){
  for (int t = 0; t < Nt; t++) {
    for (int x = 0; x < Nx; x++) {
      for (int y = 0; y < Ny; y++) {
        for (int z = 0; z < Nz; z++) {
          ix = idx.site(x, y, z, t);

          for (int mu = 0; mu < 4; mu++) {
            Tmp = U->mat(ix, mu);
            for (int c = 0; c < Nc * Nc; ++c) {
              vout.general(m_vl, "%d %d %d %d %d %d %0.16e %0.16e\n",
                           x, y, z, t, mu, c, Tmp.r(c), Tmp.i(c));
            }
          }
        }
      }
    }
  }
}


//====================================================================
//============================================================END=====
