/*!
        @file    $Id:: action_G_Plaq_SF.h #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef ACTION_G_PLAQ_SF_INCLUDED
#define ACTION_G_PLAQ_SF_INCLUDED

#include "action.h"
#include "staples_SF.h"

#include "bridgeIO.h"
using Bridge::vout;

//- parameters class
class Parameters_Action_G_Plaq_SF : virtual public Parameters
{
 public:
  Parameters_Action_G_Plaq_SF();
};
//- end

/*!
  \brief HMC action class for plaquette gauge action with SF BC.

  SF boundary condition is intrduced accrdong to the following policy.
  <ul>
    <li>The total size of temporal links is Nt with the Dirichlet BC.
    <li>We need t=0, ..., Nt sites for Nt links.
    <li>The boundary spatial link at t=0 is dummy.
    <ul>
      <li>Calculation of the Hamiltonian and force is overridden to given the correct result without using the spatial link at t=0 stored in the Field_G object.
      <li>The spatial link at t=0 is updated by a randomly given conjugate momentum but is not used for the Hamiltoniand nor the force.
      <li>The corresponding force for the spatial conjugate momentum at t=0 is set to zero. This conjugate momentum is not updated at the boundary.
    </ul>
    <li>The boundary spatial sites at t=Nt does not exist in the Field_G object.
    <ul>
      <li>Calculation of the Hamiltonian and force is overridden to given the correct result using an approriate matrix for the spatial link at t=Nt.
    </ul>
  </ul>
  The plaquette gauge action is given by
\f[
S_g[U]=\frac{\beta}{N_c}\sum_{x}\sum_{\mu<\nu}\omega_{\mu\nu}(x)
{\rm Re}{\rm Tr}\left(-U_{\mu\nu}(x)\right)
\f]
\f$\omega_{0i}(t=0)=\omega_{0i}(t=T)=c_t\f$,
\f$\omega_{\mu\nu}(n)=1\f$: otherwise.
  <ul>
  <li>A major difference from Action_G_Plaq class is a use of Staples_SF instead of Staples.
  <ul>
  <li>For correct Hamltonian with SF BC.
  <li>For correct force with SF BC.
  </ul>
  <li>Boundary condition can be accessed with m_phi and m_phipr.
  <li>Boundary improvement factor is stored in m_ct.
  <li> [25 Jan. 2012 Y.Taniguchi]
  </ul>
    (Coding history will be recovered from trac.)
    YAML is implemented.     [14 Nov 2012 Y.Namekawa]
*/

class Action_G_Plaq_SF : public Action
{
 private:
  double m_beta;

  //! SF boundary condition at t=0
  double *m_phi;
  //! SF boundary condition at t=Nt
  double *m_phipr;
  //! SF boundary improvement coefficient for the plaquatte action
  double m_ct;

  std::string m_label;
  Field       m_force;

  Field_G    *m_U;
  int        m_status_linkv;
  Staples_SF m_staple;

  RandomNumbers *m_rand;

 public:
  Action_G_Plaq_SF()
    : Action() {}

  ~Action_G_Plaq_SF() {}

  void set_parameters(const Parameters& params);
  void set_parameters(double beta, double *phi, double *phipr, double ct);

  void set_label(std::string label)
  {
    m_label = label;
    vout.detailed(m_vl, "  label: %s\n", m_label.c_str());
  }

  std::string get_label()
  {
    return m_label;
  }

  void set_config(Field *U)
  {
    m_U = (Field_G *)U;
  }

  void notify_linkv()
  {
    m_status_linkv = 0;
  }

  double langevin(RandomNumbers *);

  double calcH();

  const Field force();

  /*!
    \brief Print out the gauge force for the boundary spatial links (for debugging).
  */
  void print_force(const Field_G *);
};
#endif
