/*!
        @file    $Id:: action_G_Rectangle.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "action_G_Rectangle.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("beta", 0.0);
    param.Register_double("c_plaq", 0.0);
    param.Register_double("c_rect", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Action.G_Rectangle", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Action_G_Rectangle::Parameters_Action_G_Rectangle() { append_entry(*this); }
//- end

//====================================================================
void Action_G_Rectangle::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double beta, c_plaq, c_rect;

  int err = 0;
  err += params.fetch_double("beta", beta);
  err += params.fetch_double("c_plaq", c_plaq);
  err += params.fetch_double("c_rect", c_rect);

  if (err) {
    vout.crucial(m_vl, "Action_G_Rectangle: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(beta, c_plaq, c_rect);
}


//====================================================================
void Action_G_Rectangle::set_parameters(double beta,
                                        double c_plaq, double c_rect)
{
  //- print input parameters
  vout.general(m_vl, "Action_G_Rectangle:\n");
  vout.general(m_vl, "  beta   = %12.6f\n", beta);
  vout.general(m_vl, "  c_plaq = %12.6f\n", c_plaq);
  vout.general(m_vl, "  c_rect = %12.6f\n", c_rect);

  //- range check
  // NB. beta,c_plaq,c_rect == 0 is allowed.

  //- store values
  m_beta   = beta;
  m_c_plaq = c_plaq;
  m_c_rect = c_rect;

  //- post-process
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;
  m_force.reset(NinG, Nvol, Ndim);

  m_status_linkv = 0;
}


//====================================================================
double Action_G_Rectangle::langevin(RandomNumbers *rand)
{
  m_status_linkv = 0;

  double H_U = calcH();

  return H_U;
}


//====================================================================
double Action_G_Rectangle::calcH()
{
  int Nc   = CommonParameters::Nc();
  int Ndim = CommonParameters::Ndim();
  int Nvol = CommonParameters::Nvol();
  int Lvol = CommonParameters::Lvol();

  int Ndim2  = Ndim * (Ndim - 1) / 2;
  int size_U = Lvol * Ndim2;

  Field_G Cup1(Nvol, 1), Cup2(Nvol, 1);
  Field_G Cdn1(Nvol, 1), Cdn2(Nvol, 1);
  Field_G Umu(Nvol, 1), Unu(Nvol, 1);
  Field_G v(Nvol, 1), w(Nvol, 1), c(Nvol, 1);

  double plaqF = 0.0;
  double rectF = 0.0;

  vout.general(m_vl, "  Action_G_Rectangle: %s\n", m_label.c_str());

  for (int mu = 0; mu < Ndim; ++mu) {
    for (int nu = mu + 1; nu < Ndim; ++nu) {
      Cup1 = m_staple.upper(*m_U, mu, nu);
      Cup2 = m_staple.upper(*m_U, nu, mu);

      // plaquette term
      for (int site = 0; site < Nvol; ++site) {
        plaqF += ReTr(m_U->mat(site, mu) * Cup1.mat_dag(site));
      }

      // rectangular terms

      //      +---+---+
      //      |       |   term
      //      x   <---+

      Umu.setpart_ex(0, *m_U, mu);
      Unu.setpart_ex(0, *m_U, nu);

      m_shift.backward(v, Cup2, mu);
      m_shift.backward(c, Umu, nu);

      w.mult_Field_Gnd(0, c, 0, v, 0);
      c.mult_Field_Gnn(0, Unu, 0, w, 0);
      for (int site = 0; site < Nvol; ++site) {
        rectF += ReTr(m_U->mat(site, mu) * c.mat_dag(site));
      }

      //      +---+
      //      |   |
      //      +   +   term
      //      |   |
      //      x   v

      m_shift.backward(v, Unu, mu);
      m_shift.backward(c, Cup1, nu);

      w.mult_Field_Gnd(0, c, 0, v, 0);
      c.mult_Field_Gnn(0, Unu, 0, w, 0);
      for (int site = 0; site < Nvol; ++site) {
        rectF += ReTr(m_U->mat(site, mu) * c.mat_dag(site));
      }
    }
  }

  plaqF = Communicator::reduce_sum(plaqF);
  rectF = Communicator::reduce_sum(rectF);

  double plaq = plaqF / Nc;
  vout.general(m_vl, "    Plaquette    = %18.8f\n", plaq / size_U);

  double H_U = m_c_plaq * (Ndim2 * Lvol - plaqF / Nc)
               + m_c_rect * (Ndim2 * Lvol * 2 - rectF / Nc);

  H_U = m_beta * H_U;

  vout.general(m_vl, "    H_Grectangle = %18.8f\n", H_U);
  vout.general(m_vl, "    H_G/dof      = %18.8f\n", H_U / size_U);

  return H_U;
}


//====================================================================
const Field Action_G_Rectangle::force()
{
  if (m_status_linkv == 0) {
    int Nc   = CommonParameters::Nc();
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();

    assert(m_U->nin() == Nc * Nc * 2);
    assert(m_U->nvol() == Nvol);
    assert(m_U->nex() == Ndim);

    double   betaNc = m_beta / Nc;
    Mat_SU_N ut(Nc);

    Field_G force(Nvol, Ndim);
    Field_G force1(Nvol, 1);

    Field_G Cup1(Nvol, 1), Cup2(Nvol, 1);
    Field_G Cdn1(Nvol, 1), Cdn2(Nvol, 1);
    Field_G Umu(Nvol, 1), Unu(Nvol, 1);
    Field_G v(Nvol, 1), w(Nvol, 1), c(Nvol, 1);

    vout.general(m_vl, "  Action_G_Rectangle: %s\n", m_label.c_str());

    for (int mu = 0; mu < Ndim; ++mu) {
      force1 = 0.0;
      for (int nu = 0; nu < Ndim; ++nu) {
        if (nu == mu) continue;

        Cup1 = m_staple.upper(*m_U, mu, nu);
        Cup2 = m_staple.upper(*m_U, nu, mu);
        Cdn1 = m_staple.lower(*m_U, mu, nu);
        Cdn2 = m_staple.lower(*m_U, nu, mu);

        // plaquette term
        force1.addpart_ex(0, Cup1, 0, m_c_plaq);
        force1.addpart_ex(0, Cdn1, 0, m_c_plaq);

        // rectangular term

        Umu.setpart_ex(0, *m_U, mu);
        Unu.setpart_ex(0, *m_U, nu);

        //      +---+---+
        //      |       |   term
        //      x   <---+

        m_shift.backward(v, Cup2, mu);
        m_shift.backward(c, Umu, nu);

        w.mult_Field_Gnd(0, c, 0, v, 0);
        c.mult_Field_Gnn(0, Unu, 0, w, 0);

        force1.addpart_ex(0, c, 0, m_c_rect);

        //      +---+
        //      |   |
        //      +   +   term
        //      |   |
        //      x   v

        m_shift.backward(v, Unu, mu);
        m_shift.backward(c, Cup1, nu);

        w.mult_Field_Gnd(0, c, 0, v, 0);
        c.mult_Field_Gnn(0, Unu, 0, w, 0);

        force1.addpart_ex(0, c, 0, m_c_rect);

        //      +---+---+
        //      |       |   term
        //      +---x   v

        m_shift.backward(v, Unu, mu);
        m_shift.backward(c, Umu, nu);

        w.mult_Field_Gnd(0, c, 0, v, 0);
        c.mult_Field_Gnn(0, Cdn2, 0, w, 0);

        force1.addpart_ex(0, c, 0, m_c_rect);

        //      x   <---+
        //      |       |   term
        //      +---+---+

        m_shift.backward(v, Cup2, mu);

        w.mult_Field_Gnn(0, Umu, 0, v, 0);
        v.mult_Field_Gdn(0, Unu, 0, w, 0);

        m_shift.forward(c, v, nu);

        force1.addpart_ex(0, c, 0, m_c_rect);

        //      x   ^
        //      |   |
        //      +   +   term
        //      |   |
        //      +---+

        m_shift.backward(v, Unu, mu);

        w.mult_Field_Gnn(0, Cdn1, 0, v, 0);
        v.mult_Field_Gdn(0, Unu, 0, w, 0);

        m_shift.forward(c, v, nu);

        force1.addpart_ex(0, c, 0, m_c_rect);

        //      +---x   ^
        //      |       |   term
        //      +---+---+

        m_shift.backward(v, Unu, mu);

        w.mult_Field_Gnn(0, Umu, 0, v, 0);
        v.mult_Field_Gdn(0, Cdn2, 0, w, 0);

        m_shift.forward(c, v, nu);

        force1.addpart_ex(0, c, 0, m_c_rect);
      }

      force.mult_Field_Gnd(mu, *m_U, mu, force1, 0);
      force.at_Field_G(mu);
    }

    force *= -betaNc;

    m_force = (Field)force;
    ++m_status_linkv;

    double Fave, Fmax, Fdev;
    m_force.stat(Fave, Fmax, Fdev);
    vout.general(m_vl, "    Frectangle_ave = %12.6f  Frectangle_max = %12.6f  Frectangle_dev = %12.6f\n",
                 Fave, Fmax, Fdev);

    return m_force;
  } else {
    vout.general(m_vl, "  Action_G_Rectangle returns previous force.\n");
    return m_force;
  }
}


//====================================================================
//============================================================END=====
