/*!
        @file    $Id:: action_G_Rectangle_SF.cpp #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "action_G_Rectangle_SF.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("beta", 0.0);
    param.Register_double("c_plaq", 0.0);
    param.Register_double("c_rect", 0.0);

    param.Register_double("ct0", 0.0);
    param.Register_double("ct1", 0.0);
    param.Register_double("ct2", 0.0);

    param.Register_double("ctr0", 0.0);
    param.Register_double("ctr1", 0.0);
    param.Register_double("ctr2", 0.0);

    param.Register_double_vector("phi", std::valarray<double>());
    param.Register_double_vector("phipr", std::valarray<double>());

    // param.Register_double("ct" , 0.0);
    // param.Register_double("ctr", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Action.G_Rectangle_SF", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Action_G_Rectangle_SF::Parameters_Action_G_Rectangle_SF() { append_entry(*this); }
//- end

//====================================================================
void Action_G_Rectangle_SF::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double                beta, c_plaq, c_rect;
  double                ct0, ct1, ct2, ctr0, ctr1, ctr2;
  std::valarray<double> phi, phipr;

  int err = 0;
  err += params.fetch_double("beta", beta);
  err += params.fetch_double("c_plaq", c_plaq);
  err += params.fetch_double("c_rect", c_rect);

  err += params.fetch_double("ct0", ct0);
  err += params.fetch_double("ct1", ct1);
  err += params.fetch_double("ct2", ct2);

  err += params.fetch_double("ctr0", ctr0);
  err += params.fetch_double("ctr1", ctr1);
  err += params.fetch_double("ctr2", ctr2);

  err += params.fetch_double_vector("phi", phi);
  err += params.fetch_double_vector("phipr", phipr);

  if (err) {
    vout.crucial(m_vl, "Action_G_Rectangle_SF: fetch error, input parameter not found.\n");
    abort();
  }


  double gg = 6.0 / beta;

  double ct  = ct0 + ct1 * gg + ct2 * gg * gg;
  double ctr = ctr0 + ctr1 * gg + ctr2 * gg * gg;

  set_parameters(beta, c_plaq, c_rect, &phi[0], &phipr[0], ct, ctr);
}


//====================================================================

/*!
  Set parameters for the improved gauge action with the SF boundary.
  <ul>
    <li> m_beta
    <li> m_c_plaq, m_c_rect: plaquette and rectangle factor.
    <ul>
      <li> Iwasaki action: c_plaq =  3.648, c_rect = -0.331
    </ul>
    <li> m_phi, m_phipr: boundary spatial link
    <li> m_ct: improvement factor for the boundary temporal plaquette.
    <li> m_ctr: improvement factor for the boundary temporal rectangle with two links attached to the boundary.
  </ul>
*/
void Action_G_Rectangle_SF::set_parameters(double beta, double c_plaq, double c_rect,
                                           double *phi, double *phipr, double ct, double ctr)
{
  //- print input parameters
  vout.general(m_vl, "Action_G_Rectangle_SF:\n");
  vout.general(m_vl, "  beta   = %12.6f\n", beta);
  vout.general(m_vl, "  c_plaq = %12.6f\n", c_plaq);
  vout.general(m_vl, "  c_rect = %12.6f\n", c_rect);
  vout.general(m_vl, "  phi1   = %12.6f\n", phi[0]);
  vout.general(m_vl, "  phi2   = %12.6f\n", phi[1]);
  vout.general(m_vl, "  phi3   = %12.6f\n", phi[2]);
  vout.general(m_vl, "  phipr1 = %12.6f\n", phipr[0]);
  vout.general(m_vl, "  phipr2 = %12.6f\n", phipr[1]);
  vout.general(m_vl, "  phipr3 = %12.6f\n", phipr[2]);
  vout.general(m_vl, "  ct     = %12.6f\n", ct);
  vout.general(m_vl, "  ctr    = %12.6f\n", ctr);

  //- range check
  // NB. beta,c_plaq,c_rect,phi,phipr,ct,ctr = 0 is allowed.

  //- store values
  m_beta   = beta;
  m_c_plaq = c_plaq;
  m_c_rect = c_rect;

  m_ct  = ct;
  m_ctr = ctr;
  //  m_phi   = phi  ;
  //  m_phipr = phipr;

  //- post-process
  m_staple.set_parameters(phi, phipr);

  int    Lx = CommonParameters::Lx();
  double c0r, c0i, c1r, c1i, c2r, c2i;
  c0r = cos(phi[0] / Lx);
  c0i = sin(phi[0] / Lx);
  c1r = cos(phi[1] / Lx);
  c1i = sin(phi[1] / Lx);
  c2r = cos(phi[2] / Lx);
  c2i = sin(phi[2] / Lx);
  wk.zero();
  wk.set(0, 0, c0r, c0i);
  wk.set(1, 1, c1r, c1i);
  wk.set(2, 2, c2r, c2i);

  c0r = cos(phipr[0] / Lx);
  c0i = sin(phipr[0] / Lx);
  c1r = cos(phipr[1] / Lx);
  c1i = sin(phipr[1] / Lx);
  c2r = cos(phipr[2] / Lx);
  c2i = sin(phipr[2] / Lx);
  wkpr.zero();
  wkpr.set(0, 0, c0r, c0i);
  wkpr.set(1, 1, c1r, c1i);
  wkpr.set(2, 2, c2r, c2i);


  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;
  m_force.reset(NinG, Nvol, Ndim);

  m_status_linkv = 0;
}


//====================================================================
double Action_G_Rectangle_SF::langevin(RandomNumbers *rand)
{
  m_status_linkv = 0;

  double H_U = calcH();

  return H_U;
}


//====================================================================

/*!
  The improved gauge action with rectangle term:
\f[
 S_G=-\frac{\beta}{N_c}\left(c_0\sum_p{\rm Re}{\rm Tr} U_p
 +c_1\sum_r{\rm Re}{\rm Tr} U_r\right)
\f]
  <ul>
  <li>one plaquette term is added.
  <li>Two rectangular terms are added:
  <pre>
                               +---+
           +---+---+           |   |
           |       |           +   +
           x   <---+           |   |
                               x   v
  </pre>
  <li>We use Wk, Wk' for the boundary spatial link.
  <li>Contributions from the boundary spatial plaquettes and rectangles are set to zero.
  <li>The temporal rectangle that cross the boundary is set to zero.
<pre>
     +---+
     |   |
 t=0 +   +  --> 0
     |   |
     x   v
</pre>
  <ul>
  <li>These are automaticaly expected by a property of Staples_SF::upper()
  </ul>
  <li>Tree level improvement factor ct, ctr is implemented.
<pre>
      +---+       +---+---+
   ct |   |   ctr |       |
  t=0 x---+   t=0 x---+---+

      +---+       +---+---+
   ct |   |   ctr |       |
 Nt-1 x---+  Nt-1 x---+---+
</pre>
  </ul>
 */
double Action_G_Rectangle_SF::calcH()
{
  int Ndim = CommonParameters::Ndim();
  int Nvol = CommonParameters::Nvol();
  int Lvol = CommonParameters::Lvol();

  int Nx   = CommonParameters::Nx();
  int Ny   = CommonParameters::Ny();
  int Nz   = CommonParameters::Nz();
  int Nt   = CommonParameters::Nt();
  int NPEt = CommonParameters::NPEt();

  int Ndim2  = Ndim * (Ndim - 1) / 2;
  int size_U = Lvol * Ndim2;

  //Field_G Cup1(Nvol,1), Cup2(Nvol,1);
  //Field_G Cdn1(Nvol,1), Cdn2(Nvol,1);
  //  Field_G Umu(Nvol,1), Unu(Nvol,1);
  //  Field_G v(Nvol,1), w(Nvol,1), c(Nvol,1);
  Field_G_SF Cup1, Cup2;
  Field_G_SF Cdn1, Cdn2;
  Field_G_SF Umu, Unu;
  Field_G_SF v, w, c;

  Mat_SU_N vmat(Nc), wmat(Nc), cmat(Nc);

  double plaqF = 0.0;
  double rectF = 0.0;

  vout.general(m_vl, "  Action_G_Rectangle_SF: %s\n", m_label.c_str());

  for (int mu = 0; mu < Ndim; ++mu) {
    for (int nu = mu + 1; nu < Ndim; ++nu) {
      Cup1 = m_staple.upper(*m_U, mu, nu);
      Cup2 = m_staple.upper(*m_U, nu, mu);

      // plaquette term
      Unu = Cup2;
      // If the node is at the boundary the temporal plaquette is multiplied with ct.
      if ((nu == 3) && (Communicator::ipe(3) == 0)) {
        Unu.mult_ct_boundary(0, m_ct);
      }
      if ((nu == 3) && (Communicator::ipe(3) == NPEt - 1)) {
        Unu.mult_ct_boundary(Nt - 1, m_ct);
      }
      for (int site = 0; site < Nvol; ++site) {
        plaqF += ReTr(m_U->mat(site, nu) * Unu.mat_dag(site));
      }

      // rectangular terms

      //      +---+---+
      //      |       |   term
      //      x   <---+

      Umu.setpart_ex(0, *m_U, mu);
      Unu.setpart_ex(0, *m_U, nu);
      if ((Communicator::ipe(3) == 0) && (nu == 3)) {
        Umu.set_boundary_wk(wk);
      }

      m_shift.backward(v, Cup2, mu);
      m_shift.backward(c, Umu, nu);

      if ((Communicator::ipe(3) == 0) && (nu == 3)) {
        c.mult_ct_boundary(0, m_ctr);
      }
      if ((Communicator::ipe(3) == NPEt - 1) && (nu == 3)) {
        c.set_boundary_wkpr(wkpr);
        c.mult_ct_boundary(Nt - 1, m_ctr);
      }

      w.mult_Field_Gnd(0, c, 0, v, 0);
      c.mult_Field_Gnn(0, Unu, 0, w, 0);
      for (int site = 0; site < Nvol; ++site) {
        rectF += ReTr(Umu.mat(site) * c.mat_dag(site));
      }

      //      +---+
      //      |   |
      //      +   +   term
      //      |   |
      //      x   v

      m_shift.backward(v, Unu, mu);
      m_shift.backward(c, Cup1, nu);

      w.mult_Field_Gnd(0, c, 0, v, 0);
      c.mult_Field_Gnn(0, Unu, 0, w, 0);
      for (int site = 0; site < Nvol; ++site) {
        rectF += ReTr(Umu.mat(site) * c.mat_dag(site));
      }
    }
  }

  plaqF = Communicator::reduce_sum(plaqF);
  rectF = Communicator::reduce_sum(rectF);

  //  double plaq = plaqF/Nc;
  //  vout.general(m_vl,"    Plaquette    = %18.8f\n",plaq/size_U);

  //  double H_U =  m_c_plaq * (Ndim2*Lvol - plaqF/Nc)
  //              + m_c_rect * (Ndim2*Lvol*2 - rectF/Nc);
  double H_U = m_c_plaq * (-plaqF / Nc)
               + m_c_rect * (-rectF / Nc);

  H_U = m_beta * H_U;

  vout.general(m_vl, "    H_Grectangle = %18.8f\n", H_U);
  //  vout.general(m_vl,"    H_G/dof      = %18.8f\n",H_U/size_U);

  return H_U;
}


//====================================================================

/*!
  The force for the rectangle improved gauge action with the SF boundary.
  <ul>
  <li>We use Wk, Wk' for the boundary spatial link.
  <li>The boundary improvement factor ct, ctr is implemented.
  <li>ctr is multiplied to the following temporal rectangle staple
  <pre>
        +---+---+        +---+---+
        |  ctr              ctr  |
    t=0 +---+---x        x---+---+

        x   <---+        +---x   ^
        |  ctr  |        |  ctr  |
    t=0 +---+---+        +---+---+

        +---+---+        +---+---+
        |  ctr              ctr  |
 t=Nt-1 +---+---x        x---+---+

        +---+---+        +---+---+
        |  ctr  |        |  ctr  |
 t=Nt-1 x   <---+        +---x   v
  </pre>
  <li>Force for the boundary spatial link is set to zero.
  <pre>
        +---+---+             +---+---+
        |       |  --> 0      |       |  --> 0
    t=0 x   <---+         t=0 +---x   v

    t=0 x   <---+         t=0 +---x   ^
        |       |  --> 0      |       |  --> 0
        +---+---+             +---+---+
  </pre>
  <ul>
  <li>We notice that the upper and lower staple accompanied with the boundary spatial link is set to zero by Staples_SF::upper() and Staples_SF::lower().
  Corresponding contributions to the boundary spatial link are automaticaly zero.
  </ul>
  <li>Contribution from the non existing rectangle is automatically zero by Staples_SF::upper() and Staples_SF::lower().
  <pre>
        +---+        +---+
            |        |
    t=0 ^   +    t=0 +   ^  --> 0
        |   |        |   |
        +---+        +---+

        +---+        +---+
        |   |        |   |
   t=Nt +   +   t=Nt +   +  --> 0
            |        |
        <---+        +--->
  </pre>

  </ul>
*/
const Field Action_G_Rectangle_SF::force()
{
  if (m_status_linkv == 0) {
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();

    int Nx   = CommonParameters::Nx();
    int Ny   = CommonParameters::Ny();
    int Nz   = CommonParameters::Nz();
    int Nt   = CommonParameters::Nt();
    int NPEt = CommonParameters::NPEt();

    assert(m_U->nin() == Nc * Nc * 2);
    assert(m_U->nvol() == Nvol);
    assert(m_U->nex() == Ndim);

    double   betaNc = m_beta / Nc;
    Mat_SU_N ut(Nc);

    Field_G force(Nvol, Ndim);
    Field_G force1(Nvol, 1);

    //   Field_G Cup1(Nvol,1), Cup2(Nvol,1);
    //   Field_G Cdn1(Nvol,1), Cdn2(Nvol,1);
    //   Field_G Umu(Nvol,1), Unu(Nvol,1);
    //   Field_G v(Nvol,1), w(Nvol,1), c(Nvol,1);
    Field_G_SF Cup1, Cup2;
    Field_G_SF Cdn1, Cdn2;
    Field_G_SF Umu, Unu;
    Field_G_SF v, w, c;

    Mat_SU_N vmat(Nc), wmat(Nc), cmat(Nc);
    Mat_SU_N wzero(Nc);
    wzero.zero();

    vout.general(m_vl, "  Action_G_Rectangle_SF: %s\n", m_label.c_str());

    for (int mu = 0; mu < Ndim; ++mu) {
      force1 = 0.0;
      for (int nu = 0; nu < Ndim; ++nu) {
        if (nu == mu) continue;

        Cup1 = m_staple.upper(*m_U, mu, nu);
        Cup2 = m_staple.upper(*m_U, nu, mu);
        Cdn1 = m_staple.lower(*m_U, mu, nu);
        Cdn2 = m_staple.lower(*m_U, nu, mu);

        // plaquette term
        Umu = Cup1;
        Unu = Cdn1;
        if (Communicator::ipe(3) == 0) {
          if (mu == 3) {
            Umu.mult_ct_boundary(0, m_ct);
            Unu.mult_ct_boundary(0, m_ct);
          }
          if (nu == 3) {
            Unu.mult_ct_boundary(1, m_ct);
          }
        }
        if (Communicator::ipe(3) == NPEt - 1) {
          if (mu == 3) {
            Umu.mult_ct_boundary(Nt - 1, m_ct);
            Unu.mult_ct_boundary(Nt - 1, m_ct);
          }
          if (nu == 3) {
            Umu.mult_ct_boundary(Nt - 1, m_ct);
          }
        }
        force1.addpart_ex(0, Umu, 0, m_c_plaq);
        force1.addpart_ex(0, Unu, 0, m_c_plaq);

        // rectangular term
        Umu.setpart_ex(0, *m_U, mu);
        Unu.setpart_ex(0, *m_U, nu);
        // For the boundary spatial link, use Wk.
        if ((Communicator::ipe(3) == 0) && (nu == 3)) Umu.set_boundary_wk(wk);
        if ((Communicator::ipe(3) == 0) && (mu == 3)) Unu.set_boundary_wk(wk);

        //      +---+---+
        //      |       |   term
        //      x   <---+

        m_shift.backward(v, Cup2, mu);
        m_shift.backward(c, Umu, nu);
        // The force for the spatial link near the boundary. Multiplied with ctr.
        if ((Communicator::ipe(3) == NPEt - 1) && (nu == 3)) {
          c.set_boundary_wkpr(wkpr);
          c.mult_ct_boundary(Nt - 1, m_ctr);
        }
        w.mult_Field_Gnd(0, c, 0, v, 0);
        c.mult_Field_Gnn(0, Unu, 0, w, 0);
        // The force for the boundary spatial link is set to zero.
        if ((Communicator::ipe(3) == 0) && (nu == 3)) c.set_boundary_zero();
        force1.addpart_ex(0, c, 0, m_c_rect);

        //      +---+
        //      |   |
        //      +   +   term
        //      |   |
        //      x   v

        m_shift.backward(v, Unu, mu);
        m_shift.backward(c, Cup1, nu);
        // The force for the temporal link near the boundary. Multiplied with ctr.
        if ((Communicator::ipe(3) == NPEt - 1) && (mu == 3)) {
          v.set_boundary_wkpr(wkpr);
          v.mult_ct_boundary(Nt - 1, m_ctr);
        }
        w.mult_Field_Gnd(0, c, 0, v, 0);
        c.mult_Field_Gnn(0, Unu, 0, w, 0);
        // The force for the boundary spatial link is set to zero.
        if ((Communicator::ipe(3) == 0) && (nu == 3)) c.set_boundary_zero();
        // The force for the boundary temporal link.
        if ((Communicator::ipe(3) == 0) && (mu == 3)) c.mult_ct_boundary(0, m_ctr);
        force1.addpart_ex(0, c, 0, m_c_rect);

        //      +---+---+
        //      |       |   term
        //      +---x   v

        m_shift.backward(v, Unu, mu);
        m_shift.backward(c, Umu, nu);
        if ((Communicator::ipe(3) == NPEt - 1) && (nu == 3)) {
          c.set_boundary_wkpr(wkpr);
          c.mult_ct_boundary(Nt - 1, m_ctr);
        }
        if ((Communicator::ipe(3) == NPEt - 1) && (mu == 3)) v.set_boundary_wkpr(wkpr);
        w.mult_Field_Gnd(0, c, 0, v, 0);
        c.mult_Field_Gnn(0, Cdn2, 0, w, 0);
        if ((Communicator::ipe(3) == 0) && (nu == 3)) c.set_boundary_zero();
        force1.addpart_ex(0, c, 0, m_c_rect);

        //      x   <---+
        //      |       |   term
        //      +---+---+

        m_shift.backward(v, Cup2, mu);
        w.mult_Field_Gnn(0, Umu, 0, v, 0);
        v.mult_Field_Gdn(0, Unu, 0, w, 0);
        if ((Communicator::ipe(3) == 0) && (nu == 3)) v.mult_ct_boundary(0, m_ctr);
        m_shift.forward(c, v, nu);
        if ((Communicator::ipe(3) == 0) && (nu == 3)) c.set_boundary_zero();
        force1.addpart_ex(0, c, 0, m_c_rect);

        //      x   ^
        //      |   |
        //      +   +   term
        //      |   |
        //      +---+

        m_shift.backward(v, Unu, mu);
        if ((Communicator::ipe(3) == NPEt - 1) && (mu == 3)) {
          v.set_boundary_wkpr(wkpr);
          v.mult_ct_boundary(Nt - 1, m_ctr);
        }
        w.mult_Field_Gnn(0, Cdn1, 0, v, 0);
        v.mult_Field_Gdn(0, Unu, 0, w, 0);
        if ((Communicator::ipe(3) == 0) && (mu == 3)) v.mult_ct_boundary(0, m_ctr);
        m_shift.forward(c, v, nu);
        if ((Communicator::ipe(3) == 0) && (nu == 3)) c.set_boundary_zero();
        force1.addpart_ex(0, c, 0, m_c_rect);

        //      +---x   ^
        //      |       |   term
        //      +---+---+

        m_shift.backward(v, Unu, mu);
        if ((Communicator::ipe(3) == NPEt - 1) && (mu == 3)) v.set_boundary_wkpr(wkpr);
        w.mult_Field_Gnn(0, Umu, 0, v, 0);
        v.mult_Field_Gdn(0, Cdn2, 0, w, 0);
        if ((Communicator::ipe(3) == 0) && (nu == 3)) v.mult_ct_boundary(0, m_ctr);
        m_shift.forward(c, v, nu);
        if ((Communicator::ipe(3) == 0) && (nu == 3)) c.set_boundary_zero();
        force1.addpart_ex(0, c, 0, m_c_rect);
      }

      force.mult_Field_Gnd(mu, *m_U, mu, force1, 0);
      force.at_Field_G(mu);
    }

    force *= -betaNc;

    m_force = (Field)force;
    ++m_status_linkv;

    double Fave, Fmax, Fdev;
    m_force.stat(Fave, Fmax, Fdev);
    vout.general(m_vl, "    Fave = %12.6f  Fmax = %10.6f  Fdev = %12.6f\n",
                 Fave, Fmax, Fdev);

    return m_force;
  } else {
    vout.general(m_vl, "  Action_G_Rectangle_SF returns previous force.\n");
    return m_force;
  }
}


//====================================================================
//============================================================END=====
