/*!
        @file    $Id:: field_F_imp.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "field_F.h"

#include "bridgeIO.h"
using Bridge::vout;

using std::valarray;

// This implementation only applies to SU(3) group and Nd=4 case.
#define NC      3
#define NC2     6
#define NDF     18
#define ND      4
#define NCD     12
#define NCD2    24

//====================================================================
void Field_F::check()
{
  assert(NC == CommonParameters::Nc());
  assert(ND == CommonParameters::Nd());
}


//====================================================================
void Field_F::mult_GM(const GammaMatrix& gm, const Field_F& x)
{
  assert(x.nex() == m_Nex);
  assert(x.nvol() == m_Nvol);

  int    id[ND];
  int    idc_r[ND];
  int    idc_i[ND];
  double gv_r[ND];
  double gv_i[ND];

  for (int s = 0; s < ND; ++s) {
    id[s]    = gm.index(s);
    gv_r[s]  = gm.value_r(s);
    gv_i[s]  = gm.value_i(s);
    idc_r[s] = gm.index_c(s);
    idc_i[s] = 1 - idc_r[s];
  }

  double *w;
  w = const_cast<Field_F *>(&x)->ptr(0);

  for (int ex = 0; ex < m_Nex; ++ex) {
    for (int site = 0; site < m_Nvol; ++site) {
      int iv = NCD2 * (site + m_Nvol * ex);
      for (int s = 0; s < ND; ++s) {
        int iv2 = s * NC2 + iv;
        int iw2 = id[s] * NC2 + iv;
        for (int ic = 0; ic < NC; ++ic) {
          field[2 * ic + iv2]     = gv_r[s] * w[2 * ic + idc_r[s] + iw2];
          field[2 * ic + 1 + iv2] = gv_i[s] * w[2 * ic + idc_i[s] + iw2];
        }
      }
    }
  }
}


//====================================================================
void Field_F::mult_iGM(const GammaMatrix& gm, const Field_F& x)
{
  assert(x.nex() == m_Nex);
  assert(x.nvol() == m_Nvol);

  int    id[ND];
  int    idc_r[ND];
  int    idc_i[ND];
  double gv_r[ND];
  double gv_i[ND];

  for (int s = 0; s < ND; ++s) {
    id[s]    = gm.index(s);
    gv_r[s]  = -gm.value_i(s);
    gv_i[s]  = gm.value_r(s);
    idc_r[s] = 1 - gm.index_c(s);
    idc_i[s] = 1 - idc_r[s];
  }

  double *w;
  w = const_cast<Field_F *>(&x)->ptr(0);

  for (int ex = 0; ex < m_Nex; ++ex) {
    for (int site = 0; site < m_Nvol; ++site) {
      int iv = NCD2 * (site + m_Nvol * ex);
      for (int s = 0; s < ND; ++s) {
        int iv2 = s * NC2 + iv;
        int iw2 = id[s] * NC2 + iv;
        for (int ic = 0; ic < NC; ++ic) {
          field[2 * ic + iv2]     = gv_r[s] * w[2 * ic + idc_r[s] + iw2];
          field[2 * ic + 1 + iv2] = gv_i[s] * w[2 * ic + idc_i[s] + iw2];
        }
      }
    }
  }
}


//====================================================================
void Field_F::mult_GMproj(const int pm, const GammaMatrix& gm,
                          const Field_F& w)
{
  assert(w.nvol() == m_Nvol);
  assert(w.nex() == m_Nex);

  Field_F v(w.nvol(), w.nex());
  v.mult_GM(gm, w);

  int size = m_Nin * m_Nvol * m_Nex;
  if (pm == 1) {
    for (int i = 0; i < size; ++i) {
      field[i] = 0.5 * (w.field[i] + v.field[i]);
    }
  } else if (pm == -1) {
    for (int i = 0; i < size; ++i) {
      field[i] = 0.5 * (w.field[i] - v.field[i]);
    }
  } else {
    vout.general(m_vl, "Field_F: wrong pm = %d\n", pm);
    abort();
  }
}


//====================================================================
void Field_F::mult_GMproj2(const int pm, const GammaMatrix& gm,
                           const Field_F& w)
{
  assert(w.nvol() == m_Nvol);
  assert(w.nex() == m_Nex);

  Field_F v(w.nvol(), w.nex());
  v.mult_GM(gm, w);

  int size = m_Nin * m_Nvol * m_Nex;
  if (pm == 1) {
    for (int i = 0; i < size; ++i) {
      field[i] = w.field[i] + v.field[i];
    }
  } else if (pm == -1) {
    for (int i = 0; i < size; ++i) {
      field[i] = w.field[i] - v.field[i];
    }
  } else {
    vout.general(m_vl, "Field_F: wrong pm = %d\n", pm);
    abort();
  }
}


//====================================================================
void Field_F::mult_Field_Gn(int ex, const Field_G& U, int ex1,
                            const Field_F& x, int ex2)
{
  assert(ex <= m_Nex);
  assert(ex1 <= U.nex());
  assert(ex2 <= x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  double *w;
  double *g;
  w = const_cast<Field_F *>(&x)->ptr(0);
  g = const_cast<Field_G *>(&U)->ptr(0);

  for (int site = 0; site < m_Nvol; ++site) {
    int ig = NDF * (site + m_Nvol * ex1);
    int iw = NCD2 * (site + m_Nvol * ex2);
    int iv = NCD2 * (site + m_Nvol * ex);
    for (int s = 0; s < ND; ++s) {
      for (int ic = 0; ic < NC; ++ic) {
        int ig2 = ic * NC2 + ig;
        int iw2 = s * NC2 + iw;
        int iv2 = s * NC2 + iv;
        field[2 * ic + iv2] =
          g[0 + ig2] * w[0 + iw2] - g[1 + ig2] * w[1 + iw2]
          + g[2 + ig2] * w[2 + iw2] - g[3 + ig2] * w[3 + iw2]
          + g[4 + ig2] * w[4 + iw2] - g[5 + ig2] * w[5 + iw2];
        field[2 * ic + 1 + iv2] =
          g[0 + ig2] * w[1 + iw2] + g[1 + ig2] * w[0 + iw2]
          + g[2 + ig2] * w[3 + iw2] + g[3 + ig2] * w[2 + iw2]
          + g[4 + ig2] * w[5 + iw2] + g[5 + ig2] * w[4 + iw2];
      }
    }
  }
}


//====================================================================
void Field_F::mult_Field_Gd(int ex, const Field_G& U, int ex1,
                            const Field_F& x, int ex2)
{
  assert(ex <= m_Nex);
  assert(ex1 <= U.nex());
  assert(ex2 <= x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  double *w;
  double *g;
  w = const_cast<Field_F *>(&x)->ptr(0);
  g = const_cast<Field_G *>(&U)->ptr(0);

  for (int site = 0; site < m_Nvol; ++site) {
    int ig = NDF * (site + m_Nvol * ex1);
    int iw = NCD2 * (site + m_Nvol * ex2);
    int iv = NCD2 * (site + m_Nvol * ex);
    for (int s = 0; s < ND; ++s) {
      for (int ic = 0; ic < NC; ++ic) {
        int ig2 = ic * 2 + ig;
        int iw2 = s * NC2 + iw;
        int iv2 = s * NC2 + iv;
        field[2 * ic + iv2] =
          g[0 + ig2] * w[0 + iw2] + g[1 + ig2] * w[1 + iw2]
          + g[6 + ig2] * w[2 + iw2] + g[7 + ig2] * w[3 + iw2]
          + g[12 + ig2] * w[4 + iw2] + g[13 + ig2] * w[5 + iw2];
        field[2 * ic + 1 + iv2] =
          g[0 + ig2] * w[1 + iw2] - g[1 + ig2] * w[0 + iw2]
          + g[6 + ig2] * w[3 + iw2] - g[7 + ig2] * w[2 + iw2]
          + g[12 + ig2] * w[5 + iw2] - g[13 + ig2] * w[4 + iw2];
      }
    }
  }
}


//====================================================================
void Field_F::multadd_Field_Gn(int ex, const Field_G& U, int ex1,
                               const Field_F& x, int ex2, double a)
{
  assert(ex <= m_Nex);
  assert(ex1 <= U.nex());
  assert(ex2 <= x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  double *w;
  double *g;
  w = const_cast<Field_F *>(&x)->ptr(0);
  g = const_cast<Field_G *>(&U)->ptr(0);

  for (int site = 0; site < m_Nvol; ++site) {
    int ig = NDF * (site + m_Nvol * ex1);
    int iw = NCD2 * (site + m_Nvol * ex2);
    int iv = NCD2 * (site + m_Nvol * ex);
    for (int s = 0; s < ND; ++s) {
      for (int ic = 0; ic < NC; ++ic) {
        int ig2 = ic * NC2 + ig;
        int iw2 = s * NC2 + iw;
        int iv2 = s * NC2 + iv;
        field[2 * ic + iv2] += a * (
          g[0 + ig2] * w[0 + iw2] - g[1 + ig2] * w[1 + iw2]
          + g[2 + ig2] * w[2 + iw2] - g[3 + ig2] * w[3 + iw2]
          + g[4 + ig2] * w[4 + iw2] - g[5 + ig2] * w[5 + iw2]);
        field[2 * ic + 1 + iv2] += a * (
          g[0 + ig2] * w[1 + iw2] + g[1 + ig2] * w[0 + iw2]
          + g[2 + ig2] * w[3 + iw2] + g[3 + ig2] * w[2 + iw2]
          + g[4 + ig2] * w[5 + iw2] + g[5 + ig2] * w[4 + iw2]);
      }
    }
  }
}


//====================================================================
void Field_F::multadd_Field_Gd(int ex, const Field_G& U, int ex1,
                               const Field_F& x, int ex2, double a)
{
  assert(ex <= m_Nex);
  assert(ex1 <= U.nex());
  assert(ex2 <= x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  double *w;
  double *g;
  w = const_cast<Field_F *>(&x)->ptr(0);
  g = const_cast<Field_G *>(&U)->ptr(0);

  for (int site = 0; site < m_Nvol; ++site) {
    int ig = NDF * (site + m_Nvol * ex1);
    int iw = NCD2 * (site + m_Nvol * ex2);
    int iv = NCD2 * (site + m_Nvol * ex);
    for (int s = 0; s < ND; ++s) {
      for (int ic = 0; ic < NC; ++ic) {
        int ig2 = ic * 2 + ig;
        int iw2 = s * NC2 + iw;
        int iv2 = s * NC2 + iv;
        field[2 * ic + iv2] += a * (
          g[0 + ig2] * w[0 + iw2] + g[1 + ig2] * w[1 + iw2]
          + g[6 + ig2] * w[2 + iw2] + g[7 + ig2] * w[3 + iw2]
          + g[12 + ig2] * w[4 + iw2] + g[13 + ig2] * w[5 + iw2]);
        field[2 * ic + 1 + iv2] += a * (
          g[0 + ig2] * w[1 + iw2] - g[1 + ig2] * w[0 + iw2]
          + g[6 + ig2] * w[3 + iw2] - g[7 + ig2] * w[2 + iw2]
          + g[12 + ig2] * w[5 + iw2] - g[13 + ig2] * w[4 + iw2]);
      }
    }
  }
}


//====================================================================
//============================================================END=====
