/*!
        @file    $Id:: field_F.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "field_F.h"
#include "index_lex.h"

#include "bridgeIO.h"
using Bridge::vout;

using std::valarray;

// This file includes implementation only applies to Nc = 3,
// namely the contraction of color index for construction of baryon.
#define  NC    3
#define  C1    0
#define  C2    1
#define  C3    2

//====================================================================
void Field_F::check()
{
  // do nothing.
}


//====================================================================
void Field_F::mult_GM(const GammaMatrix& gm, const Field_F& w)
{
  assert(w.nex() == m_Nex);
  assert(w.nvol() == m_Nvol);

  valarray<int>    id(m_Nd);
  valarray<int>    idc_r(m_Nd);
  valarray<int>    idc_i(m_Nd);
  valarray<double> gv_r(m_Nd);
  valarray<double> gv_i(m_Nd);

  for (int s = 0; s < m_Nd; ++s) {
    id[s]    = gm.index(s);
    gv_r[s]  = gm.value_r(s);
    gv_i[s]  = gm.value_i(s);
    idc_r[s] = gm.index_c(s);
    idc_i[s] = 1 - idc_r[s];
  }

  for (int ex = 0; ex < m_Nex; ++ex) {
    for (int site = 0; site < m_Nvol; ++site) {
      for (int s = 0; s < m_Nd; ++s) {
        for (int c = 0; c < m_Nc; ++c) {
          field[myindex(2 * c, s, site, ex)]
            = gv_r[s] * w.field[myindex(2 * c + idc_r[s], id[s], site, ex)];
          field[myindex(2 * c + 1, s, site, ex)]
            = gv_i[s] * w.field[myindex(2 * c + idc_i[s], id[s], site, ex)];
        }
      }
    }
  }
}


//====================================================================
void Field_F::mult_iGM(const GammaMatrix& gm, const Field_F& w)
{
  assert(w.nex() == m_Nex);
  assert(w.nvol() == m_Nvol);

  valarray<int>    id(m_Nd);
  valarray<int>    idc_r(m_Nd);
  valarray<int>    idc_i(m_Nd);
  valarray<double> gv_r(m_Nd);
  valarray<double> gv_i(m_Nd);

  for (int s = 0; s < m_Nd; ++s) {
    id[s]    = gm.index(s);
    gv_r[s]  = -gm.value_i(s);
    gv_i[s]  = gm.value_r(s);
    idc_r[s] = 1 - gm.index_c(s);
    idc_i[s] = 1 - idc_r[s];
  }

  for (int ex = 0; ex < m_Nex; ++ex) {
    for (int site = 0; site < m_Nvol; ++site) {
      for (int s = 0; s < m_Nd; ++s) {
        for (int c = 0; c < m_Nc; ++c) {
          field[myindex(2 * c, s, site, ex)]
            = gv_r[s] * w.field[myindex(2 * c + idc_r[s], id[s], site, ex)];
          field[myindex(2 * c + 1, s, site, ex)]
            = gv_i[s] * w.field[myindex(2 * c + idc_i[s], id[s], site, ex)];
        }
      }
    }
  }
}


//====================================================================
void Field_F::mult_GMproj(const int pm, const GammaMatrix& gm,
                          const Field_F& w)
{
  assert(w.nvol() == m_Nvol);
  assert(w.nex() == m_Nex);

  Field_F v(w.nvol(), w.nex());
  v.mult_GM(gm, w);

  int size = m_Nin * m_Nvol * m_Nex;
  if (pm == 1) {
    for (int i = 0; i < size; ++i) {
      field[i] = 0.5 * (w.field[i] + v.field[i]);
    }
  } else if (pm == -1) {
    for (int i = 0; i < size; ++i) {
      field[i] = 0.5 * (w.field[i] - v.field[i]);
    }
  } else {
    vout.general(m_vl, "Field_F: wrong pm = %d\n", pm);
    abort();
  }
}


//====================================================================
void Field_F::mult_GMproj2(const int pm, const GammaMatrix& gm,
                           const Field_F& w)
{
  assert(w.nvol() == m_Nvol);
  assert(w.nex() == m_Nex);

  Field_F v(w.nvol(), w.nex());
  v.mult_GM(gm, w);

  int size = m_Nin * m_Nvol * m_Nex;
  if (pm == 1) {
    for (int i = 0; i < size; ++i) {
      field[i] = w.field[i] + v.field[i];
    }
  } else if (pm == -1) {
    for (int i = 0; i < size; ++i) {
      field[i] = w.field[i] - v.field[i];
    }
  } else {
    vout.general(m_vl, "Field_F: wrong pm = %d\n", pm);
    abort();
  }
}


//====================================================================
void Field_F::mult_Field_Gn(int ex, const Field_G& U, int ex1,
                            const Field_F& x, int ex2)
{
  assert(ex < m_Nex);
  assert(ex1 < U.nex());
  assert(ex2 < x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  Vec_SU_N vec(m_Nc);

  for (int site = 0; site < m_Nvol; ++site) {
    for (int s = 0; s < m_Nd; ++s) {
      vec = U.mat(site, ex1) * x.vec(s, site, ex2);
      for (int cc = 0; cc < m_Nc; ++cc) {
        field[myindex(2 * cc, s, site, ex)]     = vec.r(cc);
        field[myindex(2 * cc + 1, s, site, ex)] = vec.i(cc);
      }
    }
  }
}


//====================================================================
void Field_F::mult_Field_Gd(int ex, const Field_G& U, int ex1,
                            const Field_F& x, int ex2)
{
  assert(ex < m_Nex);
  assert(ex1 < U.nex());
  assert(ex2 < x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  Vec_SU_N vec(m_Nc);

  for (int site = 0; site < m_Nvol; ++site) {
    for (int s = 0; s < m_Nd; ++s) {
      vec = U.mat_dag(site, ex1) * x.vec(s, site, ex2);
      for (int cc = 0; cc < m_Nc; ++cc) {
        field[myindex(2 * cc, s, site, ex)]     = vec.r(cc);
        field[myindex(2 * cc + 1, s, site, ex)] = vec.i(cc);
      }
    }
  }
}


//====================================================================
void Field_F::multadd_Field_Gn(int ex, const Field_G& U, int ex1,
                               const Field_F& x, int ex2, double a)
{
  assert(ex < m_Nex);
  assert(ex1 < U.nex());
  assert(ex2 < x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  Vec_SU_N vec(m_Nc);

  for (int site = 0; site < m_Nvol; ++site) {
    for (int s = 0; s < m_Nd; ++s) {
      vec = U.mat(site, ex1) * x.vec(s, site, ex2);
      for (int cc = 0; cc < m_Nc; ++cc) {
        field[myindex(2 * cc, s, site, ex)]     += a * vec.r(cc);
        field[myindex(2 * cc + 1, s, site, ex)] += a * vec.i(cc);
      }
    }
  }
}


//====================================================================
void Field_F::multadd_Field_Gd(int ex, const Field_G& U, int ex1,
                               const Field_F& x, int ex2, double a)
{
  assert(ex < m_Nex);
  assert(ex1 < U.nex());
  assert(ex2 < x.nex());
  assert(U.nvol() == m_Nvol);
  assert(x.nvol() == m_Nvol);

  Vec_SU_N vec(m_Nc);

  for (int site = 0; site < m_Nvol; ++site) {
    for (int s = 0; s < m_Nd; ++s) {
      vec = U.mat_dag(site, ex1) * x.vec(s, site, ex2);
      for (int cc = 0; cc < m_Nc; ++cc) {
        field[myindex(2 * cc, s, site, ex)]     += a * vec.r(cc);
        field[myindex(2 * cc + 1, s, site, ex)] += a * vec.i(cc);
      }
    }
  }
}


//====================================================================
//============================================================END=====
