/*!
        @file    $Id:: shiftField_lex.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/


#include "shiftField_lex.h"


//====================================================================
void ShiftField_lex::backward(Field& v, const Field& w,
                              const int mu)
{
  int bc = 1;

  if (mu == 0) {  // x-direction
    up_x(&v, &w, bc);
  } else if (mu == 1) {
    up_y(&v, &w, bc);
  } else if (mu == 2) {
    up_z(&v, &w, bc);
  } else if (mu == 3) {
    up_t(&v, &w, bc);
  } else {
    std::cout << "wrong parameter: ShiftField_lex\n";
    abort();
  }
}


//====================================================================
void ShiftField_lex::forward(Field& v, const Field& w,
                             const int mu)
{
  int bc = 1;

  if (mu == 0) {
    dn_x(&v, &w, bc);
  } else if (mu == 1) {
    dn_y(&v, &w, bc);
  } else if (mu == 2) {
    dn_z(&v, &w, bc);
  } else if (mu == 3) {
    dn_t(&v, &w, bc);
  } else {
    std::cout << "wrong parameter: ShiftField_lex\n";
    abort();
  }
}


//====================================================================
void ShiftField_lex::backward(Field& v, const Field& w,
                              const int bc, const int mu)
{
  if (mu == 0) {  // x-direction
    up_x(&v, &w, bc);
  } else if (mu == 1) {
    up_y(&v, &w, bc);
  } else if (mu == 2) {
    up_z(&v, &w, bc);
  } else if (mu == 3) {
    up_t(&v, &w, bc);
  } else {
    std::cout << "wrong parameter: ShiftField_lex\n";
    abort();
  }
}


//====================================================================
void ShiftField_lex::forward(Field& v, const Field& w,
                             const int bc, const int mu)
{
  if (mu == 0) {
    dn_x(&v, &w, bc);
  } else if (mu == 1) {
    dn_y(&v, &w, bc);
  } else if (mu == 2) {
    dn_z(&v, &w, bc);
  } else if (mu == 3) {
    dn_t(&v, &w, bc);
  } else {
    std::cout << "wrong parameter: ShiftField_lex\n";
    abort();
  }
}


//====================================================================
void ShiftField_lex::up_x(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(0) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Nx, Nex);
  Field *vt = new Field(Nin, Nvol / Nx, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        int s2 = y + Ny * (z + Nz * t);
        // bulk
        for (int x = 0; x < Nx - 1; x++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x + 1, y, z, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }
        // boundary (x=Nx-1)
        int ix = idx.site(Nx - 1, y, z, t);
        int px = idx.site(0, y, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt->set(in, s2, ex, bc2 * w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nx) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 0, 1, 0);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        int s2 = y + Ny * (z + Nz * t);
        // boundary (x=Nx-1)
        int ix = idx.site(Nx - 1, y, z, t);
        int px = idx.site(0, y, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v->set(in, ix, ex, vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::up_y(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(1) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Ny, Nex);
  Field *vt = new Field(Nin, Nvol / Ny, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (z + Nz * t);

        // bulk
        for (int y = 0; y < Ny - 1; y++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x, y + 1, z, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }

        // boundary (y=Ny-1)
        int ix = idx.site(x, Ny - 1, z, t);
        int px = idx.site(x, 0, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt->set(in, s2, ex, bc2 * w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Ny) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 1, 1, 1);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (z + Nz * t);

        // boundary (y=Ny-1)
        int ix = idx.site(x, Ny - 1, z, t);
        int px = idx.site(x, 0, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v->set(in, ix, ex, vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::up_z(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(2) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Nz, Nex);
  Field *vt = new Field(Nin, Nvol / Nz, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * t);

        // bulk
        for (int z = 0; z < Nz - 1; z++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x, y, z + 1, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }

        // boundary (z=Nz-1)
        int ix = idx.site(x, y, Nz - 1, t);
        int px = idx.site(x, y, 0, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt->set(in, s2, ex, bc2 * w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nz) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 2, 1, 2);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * t);

        // boundary (z=Nz-1)
        int ix = idx.site(x, y, Nz - 1, t);
        int px = idx.site(x, y, 0, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v->set(in, ix, ex, vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::up_t(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(3) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Nt, Nex);
  Field *vt = new Field(Nin, Nvol / Nt, Nex);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * z);

        // bulk
        for (int t = 0; t < Nt - 1; t++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x, y, z, t + 1);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }

        // boundary (t=Nt-1)
        int ix = idx.site(x, y, z, Nt - 1);
        int px = idx.site(x, y, z, 0);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            wt->set(in, s2, ex, bc2 * w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nt) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 3, 1, 3);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * z);

        // boundary (t=Nt-1)
        int ix = idx.site(x, y, z, Nt - 1);
        int px = idx.site(x, y, z, 0);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            v->set(in, ix, ex, vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::dn_x(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(0) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Nx, Nex);
  Field *vt = new Field(Nin, Nvol / Nx, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        int s2 = y + Ny * (z + Nz * t);

        // bulk
        for (int x = 1; x < Nx; x++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x - 1, y, z, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }
        // boundary (x=0)
        //int ix = idx.site(0   ,y,z,t);
        int px = idx.site(Nx - 1, y, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt->set(in, s2, ex, w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nx) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 0, -1, 4);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        int s2 = y + Ny * (z + Nz * t);
        // boundary (x=0)
        int ix = idx.site(0, y, z, t);
        //int px = idx.site(Nx-1,y,z,t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v->set(in, ix, ex, bc2 * vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::dn_y(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(1) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Ny, Nex);
  Field *vt = new Field(Nin, Nvol / Ny, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (z + Nz * t);

        // bulk
        for (int y = 1; y < Ny; y++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x, y - 1, z, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }
        // boundary (y=0)
        int ix = idx.site(x, 0, z, t);
        int px = idx.site(x, Ny - 1, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt->set(in, s2, ex, w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Ny) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 1, -1, 5);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (z + Nz * t);

        // boundary (y=0)
        int ix = idx.site(x, 0, z, t);
        int px = idx.site(x, Ny - 1, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v->set(in, ix, ex, bc2 * vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::dn_z(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(2) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Nz, Nex);
  Field *vt = new Field(Nin, Nvol / Nz, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * t);

        // bulk
        for (int z = 1; z < Nz; z++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x, y, z - 1, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }
        // boundary (z=0)
        int ix = idx.site(x, y, 0, t);
        int px = idx.site(x, y, Nz - 1, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt->set(in, s2, ex, w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nz) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 2, -1, 6);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * t);

        // boundary (z=0)
        int ix = idx.site(x, y, 0, t);
        int px = idx.site(x, y, Nz - 1, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v->set(in, ix, ex, bc2 * vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
void ShiftField_lex::dn_t(Field *v, const Field *w, const int bc)
{
  double bc2;

  if (Communicator::ipe(3) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w->nin();
  int Nex  = w->nex();
  int Nvol = w->nvol();

  Field *wt = new Field(Nin, Nvol / Nt, Nex);
  Field *vt = new Field(Nin, Nvol / Nt, Nex);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * z);

        // bulk
        for (int t = 1; t < Nt; t++) {
          int ix = idx.site(x, y, z, t);
          int px = idx.site(x, y, z, t - 1);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v->set(in, ix, ex, w->cmp(in, px, ex));
            }
          }
        }
        // boundary (t=0)
        int ix = idx.site(x, y, z, 0);
        int px = idx.site(x, y, z, Nt - 1);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            wt->set(in, s2, ex, w->cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nt) * Nex;
  Communicator::exchange(size, vt->ptr(0), wt->ptr(0), 3, -1, 7);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx; x++) {
        int s2 = x + Nx * (y + Ny * z);

        // boundary (t=0)
        int ix = idx.site(x, y, z, 0);
        int px = idx.site(x, y, z, Nt - 1);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            v->set(in, ix, ex, bc2 * vt->cmp(in, s2, ex));
          }
        }
      }
    }
  }

  delete vt;
  delete wt;
}


//====================================================================
//============================================================END=====
