/*!
        @file    $Id:: field_F_1spinor.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/


#ifndef FIELD_F_1sp_INCLUDED
#define FIELD_F_1sp_INCLUDED

#include "commonParameters.h"
#include "communicator.h"
#include "field.h"
#include "vec_SU_N.h"
using namespace SU_N;

//! Staggered-type fermion field.

/*!
   This class defines 1-spinor fermion field, which is mainly used
   by staggered-type fermions.
                                     [28 Dec 2011 H.Matsufuru]
 */
class Field_F_1spinor : public Field {
 private:
  int m_Nc;   // num of the color elements
  int m_Nc2;  // num of the double color elements
  int m_Nvol; // lattice volume
  int m_Nex;  // num of the extra indices

  int myindex(const int c2, const int site, const int ex)
  const
  {
    return Field::myindex(c2, site, ex);
  }

 public:
  Field_F_1spinor(const int Nvol = CommonParameters::Nvol(), const int Nex = 1) :
    Field(),
    m_Nc(CommonParameters::Nc()),
    m_Nvol(Nvol),
    m_Nex(Nex)
  {
    m_Nc2 = 2 * m_Nc;
    reset(m_Nc2, m_Nvol, m_Nex);
  }

  Field_F_1spinor(const Field& x) :
    Field(x),
    m_Nc(CommonParameters::Nc()),
    m_Nc2(2 * m_Nc),
    m_Nvol(x.nvol()),
    m_Nex(x.nex())
  {
  }

  double cmp_r(const int cc, const int site, const int e = 0)
  const
  {
    return field[myindex(2 * cc, site, e)];
  }

  double cmp_i(const int cc, const int site, const int e = 0)
  const
  {
    return field[myindex(2 * cc + 1, site, e)];
  }

  void set_r(const int cc, const int site, const int e,
             const double re)
  {
    field[myindex(2 * cc, site, e)] = re;
  }

  void set_i(const int cc, const int site, const int e,
             const double im)
  {
    field[myindex(2 * cc + 1, site, e)] = im;
  }

  void set_ri(const int cc, const int site, const int e,
              const double re, const double im)
  {
    field[myindex(2 * cc, site, e)]     = re;
    field[myindex(2 * cc + 1, site, e)] = im;
  }

  Vec_SU_N vec(const int site, const int e = 0) const
  {
    Vec_SU_N Tmp;

    for (int cc = 0; cc < m_Nc; ++cc) {
      Tmp.set(cc, field[myindex(2 * cc, site, e)],
              field[myindex(2 * cc + 1, site, e)]);
    }
    return Tmp;
  }

  void set_vec(const int site, const int e, const Vec_SU_N& F)
  {
    for (int cc = 0; cc < m_Nc; ++cc) {
      field[myindex(2 * cc, site, e)]     = F.r(cc);
      field[myindex(2 * cc + 1, site, e)] = F.i(cc);
    }
  }

  void add_vec(const int site, const int e, const Vec_SU_N& F)
  {
    for (int cc = 0; cc < m_Nc; ++cc) {
      field[myindex(2 * cc, site, e)]     += F.r(cc);
      field[myindex(2 * cc + 1, site, e)] += F.i(cc);
    }
  }

  void clear_vec(const int site, const int e)
  {
    for (int cc = 0; cc < m_Nc2; ++cc) {
      field[myindex(cc, site, e)] = 0.0;
    }
  }

  double operator*(const Field_F_1spinor&);

  template<typename T>
  Field_F_1spinor& operator=(const T& rhs)
  {
    *this = rhs.eval();
    return *this;
  }

  template<typename T>
  Field_F_1spinor& operator+=(const T& rhs)
  {
    *this += rhs.eval();
    return *this;
  }

  template<typename T>
  Field_F_1spinor& operator-=(const T& rhs)
  {
    *this -= rhs.eval();
    return *this;
  }

  /*
  */

  Field_F_1spinor& operator-();
  Field_F_1spinor& operator=(const double&);
  Field_F_1spinor& operator+=(const Field_F_1spinor&);
  Field_F_1spinor& operator-=(const Field_F_1spinor&);
  Field_F_1spinor& operator*=(const double&);
  Field_F_1spinor& operator*=(const dcomplex&);
  Field_F_1spinor& operator/=(const double&);
  Field_F_1spinor& operator/=(const dcomplex&);
};

inline Field_F_1spinor& Field_F_1spinor::operator=(const double& r)
{
  field = r;
  return *this;
}


inline Field_F_1spinor& Field_F_1spinor::operator-()
{
  field = -field;
  return *this;
}


inline Field_F_1spinor& Field_F_1spinor::operator+=(const Field_F_1spinor& rhs)
{
  field += rhs.field;
  return *this;
}


inline Field_F_1spinor& Field_F_1spinor::operator-=(const Field_F_1spinor& rhs)
{
  field -= rhs.field;
  return *this;
}


inline Field_F_1spinor& Field_F_1spinor::operator*=(const double& rhs)
{
  field *= rhs;
  return *this;
}


inline Field_F_1spinor& Field_F_1spinor::operator/=(const double& rhs)
{
  field /= rhs;
  return *this;
}


inline double Field_F_1spinor::operator*(const Field_F_1spinor& rhs)
{
  double a = (field * rhs.field).sum();
  double b = Communicator::reduce_sum(a);

  return b;
}
#endif
