/*!
        @file    $Id:: field_G.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/


#ifndef FIELD_G_INCLUDED
#define FIELD_G_INCLUDED

#include "bridge_complex.h"

#include "commonParameters.h"
#include "field.h"
#include "mat_SU_N.h"
using namespace SU_N;

//! SU(N) gauge field.

/*!
   This class defines SU(N) gauge field, which is used such as
   gauge configuration.
   Original version of this class was written by J.Noaki.
   H.Matsufuru added several functions and modified intefaces
   of several functionality.
                                     [28 Dec 2011 H.Matsufuru]
 */
class Field_G : public Field {
 private:
  int m_Nc;     // number of color elements
  int m_Ndf;    // number of components as real values
  int m_Nvol;   // number of sites
  int m_Nex;    // extra degree of freedom, such as mu, nu.

 public:

  Field_G(const int Nvol = CommonParameters::Nvol(), const int Nex = 1) :
    Field(),
    m_Nc(CommonParameters::Nc()),
    m_Nvol(Nvol),
    m_Nex(Nex)
  {
    m_Ndf = 2 * m_Nc * m_Nc;
    Field::reset(m_Ndf, m_Nvol, m_Nex);
  }

  Field_G(const Field& x) :
    Field(x),
    m_Nc(CommonParameters::Nc()),
    m_Ndf(x.nin()),
    m_Nvol(x.nvol()),
    m_Nex(x.nex())
  {
    assert(m_Ndf == 2 * m_Nc * m_Nc);
  }

  void reset(const int Nvol, const int Nex)
  {
    m_Nvol = Nvol;
    m_Nex  = Nex;
    Field::reset(m_Ndf, m_Nvol, m_Nex);
  }

  double cmp_r(const int cc, const int site, const int mn = 0) const
  {
    return field[myindex(2 * cc, site, mn)];
  }

  double cmp_i(const int cc, const int site, const int mn = 0) const
  {
    return field[myindex(2 * cc + 1, site, mn)];
  }

  void set_r(const int cc, const int site, const int mn, const double re)
  {
    field[myindex(2 * cc, site, mn)] = re;
  }

  void set_i(const int cc, const int site, const int mn, const double im)
  {
    field[myindex(2 * cc + 1, site, mn)] = im;
  }

  void set_ri(const int cc, const int site, const int mn,
              const double re, const double im)
  {
    field[myindex(2 * cc, site, mn)]     = re;
    field[myindex(2 * cc + 1, site, mn)] = im;
  }

  Mat_SU_N mat(const int site, const int mn = 0) const
  {
    Mat_SU_N Tmp(m_Nc);

    for (int cc = 0; cc < m_Nc * m_Nc; ++cc) {
      Tmp.set(cc, field[myindex(2 * cc, site, mn)],
              field[myindex(2 * cc + 1, site, mn)]);
    }
    return Tmp;
  }

  Mat_SU_N mat_dag(const int site, const int mn = 0) const
  {
    Mat_SU_N Tmp(m_Nc);

    for (int cc = 0; cc < m_Nc * m_Nc; ++cc) {
      Tmp.set(cc, field[myindex(2 * cc, site, mn)],
              field[myindex(2 * cc + 1, site, mn)]);
    }
    return Tmp.dag();
  }

  void mat(Mat_SU_N& Tmp, const int site, const int mn = 0) const
  {
    for (int cc = 0; cc < m_Nc * m_Nc; ++cc) {
      Tmp.set(cc, field[myindex(2 * cc, site, mn)],
              field[myindex(2 * cc + 1, site, mn)]);
    }
  }

  void mat_dag(Mat_SU_N& Tmp, const int site, const int mn = 0) const
  {
    for (int c1 = 0; c1 < m_Nc; ++c1) {
      for (int c2 = 0; c2 < m_Nc; ++c2) {
        Tmp.set(c1 + m_Nc * c2, field[myindex(2 * (c2 + m_Nc * c1), site, mn)],
                -field[myindex(2 * (c2 + m_Nc * c1) + 1, site, mn)]);
      }
    }
  }

  void set_mat(const int site, const int mn, const Mat_SU_N& U)
  {
    for (int cc = 0; cc < m_Nc * m_Nc; ++cc) {
      field[myindex(2 * cc, site, mn)]     = U.r(cc);
      field[myindex(2 * cc + 1, site, mn)] = U.i(cc);
    }
  }

  void add_mat(const int site, const int mn, const Mat_SU_N& U)
  {
    for (int cc = 0; cc < m_Nc * m_Nc; ++cc) {
      field[myindex(2 * cc, site, mn)]     += U.r(cc);
      field[myindex(2 * cc + 1, site, mn)] += U.i(cc);
    }
  }

  void add_mat(const int site, const int mn, const Mat_SU_N& U, double prf)
  {
    for (int cc = 0; cc < m_Nc * m_Nc; ++cc) {
      field[myindex(2 * cc, site, mn)]     += prf * U.r(cc);
      field[myindex(2 * cc + 1, site, mn)] += prf * U.i(cc);
    }
  }

  void xI()
  {
    for (int i = 0; i < field.size(); i += 2) {
      double real = field[i];
      field[i]     = -field[i + 1];
      field[i + 1] = real;
    }
  }

  void mult_Field_Gnn(int ex, const Field_G&, int ex1,
                      const Field_G&, int ex2);

  void mult_Field_Gnd(int ex, const Field_G&, int ex1,
                      const Field_G&, int ex2);

  void mult_Field_Gdn(int ex, const Field_G&, int ex1,
                      const Field_G&, int ex2);

  void mult_Field_Gdd(int ex, const Field_G&, int ex1,
                      const Field_G&, int ex2);

  void multadd_Field_Gnn(int ex, const Field_G&, int ex1,
                         const Field_G&, int ex2, double);

  void multadd_Field_Gnd(int ex, const Field_G&, int ex1,
                         const Field_G&, int ex2, double);

  void multadd_Field_Gdn(int ex, const Field_G&, int ex1,
                         const Field_G&, int ex2, double);

  void multadd_Field_Gdd(int ex, const Field_G&, int ex1,
                         const Field_G&, int ex2, double);

  //! bulk operation of antihermitian
  void ah_Field_G(int ex);

  //! bulk operation of antihermitian traceless
  void at_Field_G(int ex);

  Field_G& operator-();
  Field_G& operator=(const double&);
  Field_G& operator+=(const Field_G&);
  Field_G& operator-=(const Field_G&);
  Field_G& operator*=(const double&);
  Field_G& operator*=(const dcomplex&);
  Field_G& operator/=(const double&);
  Field_G& operator/=(const dcomplex&);
};

inline Field_G& Field_G::operator-()
{
  field = -field;
  return *this;
}


inline Field_G& Field_G::operator=(const double& r)
{
  field = r;
  return *this;
}


inline Field_G& Field_G::operator+=(const Field_G& rhs)
{
  field += rhs.field;
  return *this;
}


inline Field_G& Field_G::operator-=(const Field_G& rhs)
{
  field -= rhs.field;
  return *this;
}


inline Field_G& Field_G::operator*=(const double& rhs)
{
  field *= rhs;
  return *this;
}


inline Field_G& Field_G::operator/=(const double& rhs)
{
  field /= rhs;
  return *this;
}
#endif
