/*!
        @file    $Id:: index_eo.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef INDEX_EO_INCLUDED
#define INDEX_EO_INCLUDED

#include "field_G.h"
#include "communicator.h"
#include "index_lex.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Even-odd site index.

/*!
    This class defines even-odd site index.
    Only the site degree of freedom is concerned.
    Nx (x-extent inside a node) must be even in the present
    implementation.
    Coverting from and reverting to the lexical site index are
    implemented as member functions of this class.
    In present implementation, there is no superclass structure,
    and thus polymorphism is not available.
    Some of method names might be confusing; restructuring may
    be helpful.
                                      [25 Dec 2011 H.Matsufuru]
*/
class Index_eo {
 private:
  int                Nx, Ny, Nz, Nt, Nvol;
  int                Nx2, Nvol2;
  std::valarray<int> Leo;
  std::valarray<int> Site_up;
  std::valarray<int> Site_dn;
  Index_lex          m_index_lex;

  Bridge::VerboseLevel m_vl;
 public:
  Index_eo() :
    Nx(CommonParameters::Nx()),
    Ny(CommonParameters::Ny()),
    Nz(CommonParameters::Nz()),
    Nt(CommonParameters::Nt()),
    Nvol(CommonParameters::Nvol()),
    Nx2(CommonParameters::Nx() / 2),
    Nvol2(CommonParameters::Nvol() / 2),
    m_vl(CommonParameters::Vlevel())
  {
    if ((Nx % 2) == 1) {
      vout.general(m_vl, "Nx is not even: error in Index_eo.\n");
      abort();
    }

    Leo.resize(Ny * Nz * Nt);
    Site_up.resize(Nx2 * Ny * Nz * Nt * 2);
    Site_dn.resize(Nx2 * Ny * Nz * Nt * 2);

    for (int t = 0; t < Nt; ++t) {
      for (int z = 0; z < Nz; ++z) {
        for (int y = 0; y < Ny; ++y) {
          int t2 = Communicator::ipe(3) * Nt + t;
          int z2 = Communicator::ipe(2) * Nz + z;
          int y2 = Communicator::ipe(1) * Ny + y;
          Leo[y + Ny * (z + Nz * t)] = (y2 + z2 + t2) % 2;
        }
      }
    }

    for (int t = 0; t < Nt; ++t) {
      for (int z = 0; z < Nz; ++z) {
        for (int y = 0; y < Ny; ++y) {
          int yzt = y + Ny * (z + Nz * t);
          int t2  = t;
          int z2  = z;
          int y2  = y;
          for (int x2 = 0; x2 < Nx2; ++x2) {
            int s = x2 + Nx2 * (y + Ny * (z + Nz * t));
            Site_up[s]         = ((x2 + Leo[yzt]) % Nx2) + Nx2 * (y + Ny * (z + Nz * t));
            Site_up[s + Nvol2] = ((x2 + 1 - Leo[yzt]) % Nx2) + Nx2 * (y + Ny * (z + Nz * t));
            Site_dn[s]         = ((x2 - 1 + Leo[yzt] + Nx2) % Nx2) + Nx2 * (y + Ny * (z + Nz * t));
            Site_dn[s + Nvol2] = ((x2 - Leo[yzt] + Nx2) % Nx2) + Nx2 * (y + Ny * (z + Nz * t));
          }
        }
      }
    }
  }

  int leo(const int y, const int z, const int t) const
  {
    return Leo[y + Ny * (z + Nz * t)];
  }

  int site(const int x2, const int y, const int z, const int t,
           const int ieo) const
  {
    return x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;
  }

  int site(const int is, const int ieo) const
  {
    return is + Nvol2 * ieo;
  }

  int site_up(const int x2, const int y, const int z, const int t,
              const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_up[s] + Nvol2 * (1 - ieo);
  }

  int site_xup(const int x2, const int y, const int z, const int t,
               const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_up[s] + Nvol2 * (1 - ieo);
  }

  int site_dn(const int x2, const int y, const int z, const int t,
              const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_dn[s] + Nvol2 * (1 - ieo);
  }

  int site_xdn(const int x2, const int y, const int z, const int t,
               const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_dn[s] + Nvol2 * (1 - ieo);
  }

  int siteh(const int x2, const int y, const int z, const int t)
  const
  {
    return x2 + Nx2 * (y + Ny * (z + Nz * t));
  }

  int siteh_up(const int x2, const int y, const int z, const int t,
               const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_up[s];
  }

  int siteh_xup(const int x2, const int y, const int z, const int t,
                const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_up[s];
  }

  int siteh_dn(const int x2, const int y, const int z, const int t,
               const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_dn[s];
  }

  int siteh_xdn(const int x2, const int y, const int z, const int t,
                const int ieo) const
  {
    int s = x2 + Nx2 * (y + Ny * (z + Nz * t)) + Nvol2 * ieo;

    return Site_dn[s];
  }

  void convertField(Field& eo, const Field& lex);
  void convertField(Field& eo, const Field& lex, const int ieo);

  void reverseField(Field& lex, const Field& eo);
  void reverseField(Field& lex, const Field& eo, const int ieo);

  void splitField(Field& e, Field& o, const Field& eo);

  void mergeField(Field& eo, const Field& e, const Field& o);
};
#endif
