/*!
        @file    $Id:: shiftField_eo.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "shiftField_eo.h"

//====================================================================
void ShiftField_eo::backward_h(Field& v, const Field& w,
                               const int mu, const int ieo)
{
  int bc = 1;

  if (mu == 0) {        // x-direction
    up_xh(v, w, bc, ieo);
  } else if (mu == 1) { // y-direction
    up_yh(v, w, bc, ieo);
  } else if (mu == 2) { // z-direction
    up_zh(v, w, bc, ieo);
  } else if (mu == 3) { // t-direction
    up_th(v, w, bc, ieo);
  } else {
    vout.crucial(m_vl, "ShiftField_eo: wrong mu = %d\n", mu);
    abort();
  }
}


//====================================================================
void ShiftField_eo::forward_h(Field& v, const Field& w,
                              const int mu, const int ieo)
{
  int bc = 1;

  if (mu == 0) {        // x-direction
    dn_xh(v, w, bc, ieo);
  } else if (mu == 1) { // y-direction
    dn_yh(v, w, bc, ieo);
  } else if (mu == 2) { // z-direction
    dn_zh(v, w, bc, ieo);
  } else if (mu == 3) { // t-direction
    dn_th(v, w, bc, ieo);
  } else {
    vout.crucial(m_vl, "ShiftField_eo: wrong mu = %d\n", mu);
    abort();
  }
}


//====================================================================
void ShiftField_eo::backward_h(Field& v, const Field& w,
                               const int bc, const int mu, const int ieo)
{
  if (mu == 0) {        // x-direction
    up_xh(v, w, bc, ieo);
  } else if (mu == 1) { // y-direction
    up_yh(v, w, bc, ieo);
  } else if (mu == 2) { // z-direction
    up_zh(v, w, bc, ieo);
  } else if (mu == 3) { // t-direction
    up_th(v, w, bc, ieo);
  } else {
    vout.crucial(m_vl, "ShiftField_eo: wrong mu = %d\n", mu);
    abort();
  }
}


//====================================================================
void ShiftField_eo::forward_h(Field& v, const Field& w,
                              const int bc, const int mu, const int ieo)
{
  if (mu == 0) {        // x-direction
    dn_xh(v, w, bc, ieo);
  } else if (mu == 1) { // y-direction
    dn_yh(v, w, bc, ieo);
  } else if (mu == 2) { // z-direction
    dn_zh(v, w, bc, ieo);
  } else if (mu == 3) { // t-direction
    dn_th(v, w, bc, ieo);
  } else {
    vout.crucial(m_vl, "ShiftField_eo: wrong mu = %d\n", mu);
    abort();
  }
}


//====================================================================
void ShiftField_eo::backward(Field& v, const Field& w, const int mu)
{
  Index_eo index_eo;

  int Nin   = w.nin();
  int Nvol2 = w.nvol() / 2;
  int Nex   = w.nex();

  Field field_e(Nin, Nvol2, Nex);
  Field field_o(Nin, Nvol2, Nex);

  Field field_se(Nin, Nvol2, Nex);
  Field field_so(Nin, Nvol2, Nex);

  index_eo.splitField(field_e, field_o, w);

  backward_h(field_se, field_o, mu, 0);
  backward_h(field_so, field_e, mu, 1);

  index_eo.mergeField(v, field_se, field_so);
}


//====================================================================
void ShiftField_eo::forward(Field& v, const Field& w, const int mu)
{
  Index_eo index_eo;

  int Nin   = w.nin();
  int Nvol2 = w.nvol() / 2;
  int Nex   = w.nex();

  Field field_e(Nin, Nvol2, Nex);
  Field field_o(Nin, Nvol2, Nex);

  Field field_se(Nin, Nvol2, Nex);
  Field field_so(Nin, Nvol2, Nex);

  index_eo.splitField(field_e, field_o, w);

  forward_h(field_se, field_o, mu, 0);
  forward_h(field_so, field_e, mu, 1);

  index_eo.mergeField(v, field_se, field_so);
}


//====================================================================
void ShiftField_eo::backward(Field& v, const Field& w,
                             const int bc, const int mu)
{
  Index_eo index_eo;

  int Nin   = w.nin();
  int Nvol2 = w.nvol() / 2;
  int Nex   = w.nex();

  Field field_e(Nin, Nvol2, Nex);
  Field field_o(Nin, Nvol2, Nex);

  Field field_se(Nin, Nvol2, Nex);
  Field field_so(Nin, Nvol2, Nex);

  index_eo.splitField(field_e, field_o, w);

  backward_h(field_se, field_o, bc, mu, 0);
  backward_h(field_so, field_e, bc, mu, 1);

  index_eo.mergeField(v, field_se, field_so);
}


//====================================================================
void ShiftField_eo::forward(Field& v, const Field& w,
                            const int bc, const int mu)
{
  Index_eo index_eo;

  int Nin   = w.nin();
  int Nvol2 = w.nvol() / 2;
  int Nex   = w.nex();

  Field field_e(Nin, Nvol2, Nex);
  Field field_o(Nin, Nvol2, Nex);

  Field field_se(Nin, Nvol2, Nex);
  Field field_so(Nin, Nvol2, Nex);

  index_eo.splitField(field_e, field_o, w);

  forward_h(field_se, field_o, bc, mu, 0);
  forward_h(field_so, field_e, bc, mu, 1);

  index_eo.mergeField(v, field_se, field_so);
}


//====================================================================
void ShiftField_eo::up_xh(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(0) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin   = w.nin();
  int Nex   = w.nex();
  int Nvol  = w.nvol();
  int Nvol2 = (1 + Nvol / Nx2) / 2;

  Field wt(Nin, Nvol2, Nex);
  Field vt(Nin, Nvol2, Nex);

  int s2 = 0;
  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        if (idx.leo(y, z, t) == ieo) {
          // bulk
          for (int x = 0; x < Nx2; x++) {
            int ix = idx.siteh(x, y, z, t);
            for (int ex = 0; ex < Nex; ex++) {
              for (int in = 0; in < Nin; in++) {
                v.set(in, ix, ex, w.cmp(in, ix, ex));
              }
            }
          }
        } else {
          // bulk
          for (int x = 0; x < Nx2 - 1; x++) {
            int ix = idx.siteh(x, y, z, t);
            int px = idx.siteh_xup(x, y, z, t, ieo);
            for (int ex = 0; ex < Nex; ex++) {
              for (int in = 0; in < Nin; in++) {
                v.set(in, ix, ex, w.cmp(in, px, ex));
              }
            }
          }
          // boundary (x=Nx2-1)
          int px = idx.siteh(0, y, z, t);
          for (int in = 0; in < Nin; in++) {
            for (int ex = 0; ex < Nex; ex++) {
              wt.set(in, s2, ex, bc2 * w.cmp(in, px, ex));
            }
          }
          s2++;
        }
      }
    }
  }

  if (s2 > Nvol2) {
    vout.general(m_vl, "invalid size in shiftField_eo.");
    abort();
  }

  int size = Nin * Nvol2 * Nex;
  exchange(size, &vt, &wt, 0, 1, 0);

  s2 = 0;
  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        // boundary (x=Nx-1)
        if (idx.leo(y, z, t) != ieo) {
          int ix = idx.siteh(Nx2 - 1, y, z, t);
          for (int in = 0; in < Nin; in++) {
            for (int ex = 0; ex < Nex; ex++) {
              v.set(in, ix, ex, vt.cmp(in, s2, ex));
            }
          }
          s2++;
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::dn_xh(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(0) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin   = w.nin();
  int Nex   = w.nex();
  int Nvol  = w.nvol();
  int Nvol2 = (1 + Nvol / Nx2) / 2;

  Field wt(Nin, Nvol2, Nex);
  Field vt(Nin, Nvol2, Nex);

  int s2 = 0;
  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        if (idx.leo(y, z, t) == (1 - ieo)) {
          // bulk
          for (int x = 0; x < Nx2; x++) {
            int ix = idx.siteh(x, y, z, t);
            for (int ex = 0; ex < Nex; ex++) {
              for (int in = 0; in < Nin; in++) {
                v.set(in, ix, ex, w.cmp(in, ix, ex));
              }
            }
          }
        } else {
          // bulk
          for (int x = 1; x < Nx2; x++) {
            int ix = idx.siteh(x, y, z, t);
            int mx = idx.siteh_xdn(x, y, z, t, ieo);
            for (int ex = 0; ex < Nex; ex++) {
              for (int in = 0; in < Nin; in++) {
                v.set(in, ix, ex, w.cmp(in, mx, ex));
              }
            }
          }
          // boundary (x=0)
          int mx = idx.siteh(Nx2 - 1, y, z, t);
          for (int in = 0; in < Nin; in++) {
            for (int ex = 0; ex < Nex; ex++) {
              wt.set(in, s2, ex, w.cmp(in, mx, ex));
            }
          }
          s2++;
        }
      }
    }
  }

  int size = Nin * Nvol2 * Nex;
  exchange(size, &vt, &wt, 0, -1, 4);

  s2 = 0;
  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int y = 0; y < Ny; y++) {
        // boundary (x=0)
        if (idx.leo(y, z, t) != (1 - ieo)) {
          int ix = idx.siteh(0, y, z, t);
          for (int in = 0; in < Nin; in++) {
            for (int ex = 0; ex < Nex; ex++) {
              v.set(in, ix, ex, bc2 * vt.cmp(in, s2, ex));
            }
          }
          s2++;
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::up_yh(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(1) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w.nin();
  int Nex  = w.nex();
  int Nvol = w.nvol();

  Field wt(Nin, Nvol / Ny, Nex);
  Field vt(Nin, Nvol / Ny, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx2; x++) {
        // bulk
        for (int y = 0; y < Ny - 1; y++) {
          int ix = idx.siteh(x, y, z, t);
          int px = idx.siteh(x, y + 1, z, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v.set(in, ix, ex, w.cmp(in, px, ex));
            }
          }
        }

        // boundary (y=Ny-1)
        int s2 = x + Nx2 * (z + Nz * t);
        int px = idx.siteh(x, 0, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt.set(in, s2, ex, bc2 * w.cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Ny) * Nex;
  exchange(size, &vt, &wt, 1, 1, 1);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx2; x++) {
        // boundary (y=Ny-1)
        int s2 = x + Nx2 * (z + Nz * t);
        int ix = idx.siteh(x, Ny - 1, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v.set(in, ix, ex, vt.cmp(in, s2, ex));
          }
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::dn_yh(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(1) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w.nin();
  int Nex  = w.nex();
  int Nvol = w.nvol();

  Field wt(Nin, Nvol / Ny, Nex);
  Field vt(Nin, Nvol / Ny, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx2; x++) {
        // bulk
        for (int y = 1; y < Ny; y++) {
          int ix = idx.siteh(x, y, z, t);
          int px = idx.siteh(x, y - 1, z, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v.set(in, ix, ex, w.cmp(in, px, ex));
            }
          }
        }
        // boundary (y=0)
        int s2 = x + Nx2 * (z + Nz * t);
        int px = idx.siteh(x, Ny - 1, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt.set(in, s2, ex, w.cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Ny) * Nex;
  exchange(size, &vt, &wt, 1, -1, 5);

  for (int t = 0; t < Nt; t++) {
    for (int z = 0; z < Nz; z++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (z + Nz * t);

        // boundary (y=0)
        int ix = idx.siteh(x, 0, z, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v.set(in, ix, ex, bc2 * vt.cmp(in, s2, ex));
          }
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::up_zh(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(2) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w.nin();
  int Nex  = w.nex();
  int Nvol = w.nvol();

  Field wt(Nin, Nvol / Nz, Nex);
  Field vt(Nin, Nvol / Nz, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        // bulk
        for (int z = 0; z < Nz - 1; z++) {
          int ix = idx.siteh(x, y, z, t);
          int px = idx.siteh(x, y, z + 1, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v.set(in, ix, ex, w.cmp(in, px, ex));
            }
          }
        }

        // boundary (z=Nz-1)
        int s2 = x + Nx2 * (y + Ny * t);
        int px = idx.siteh(x, y, 0, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt.set(in, s2, ex, bc2 * w.cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nz) * Nex;
  exchange(size, &vt, &wt, 2, 1, 2);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * t);

        // boundary (z=Nz-1)
        int ix = idx.siteh(x, y, Nz - 1, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v.set(in, ix, ex, vt.cmp(in, s2, ex));
          }
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::dn_zh(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(2) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w.nin();
  int Nex  = w.nex();
  int Nvol = w.nvol();

  Field wt(Nin, Nvol / Nz, Nex);
  Field vt(Nin, Nvol / Nz, Nex);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * t);

        // bulk
        for (int z = 1; z < Nz; z++) {
          int ix = idx.siteh(x, y, z, t);
          int px = idx.siteh(x, y, z - 1, t);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v.set(in, ix, ex, w.cmp(in, px, ex));
            }
          }
        }
        // boundary (z=0)
        int px = idx.siteh(x, y, Nz - 1, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            wt.set(in, s2, ex, w.cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nz) * Nex;
  exchange(size, &vt, &wt, 2, -1, 6);

  for (int t = 0; t < Nt; t++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * t);

        // boundary (z=0)
        int ix = idx.siteh(x, y, 0, t);
        for (int in = 0; in < Nin; in++) {
          for (int ex = 0; ex < Nex; ex++) {
            v.set(in, ix, ex, bc2 * vt.cmp(in, s2, ex));
          }
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::up_th(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(3) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w.nin();
  int Nex  = w.nex();
  int Nvol = w.nvol();

  Field wt(Nin, Nvol / Nt, Nex);
  Field vt(Nin, Nvol / Nt, Nex);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * z);

        // bulk
        for (int t = 0; t < Nt - 1; t++) {
          int ix = idx.siteh(x, y, z, t);
          int px = idx.siteh(x, y, z, t + 1);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v.set(in, ix, ex, w.cmp(in, px, ex));
            }
          }
        }

        // boundary (t=Nt-1)
        int px = idx.siteh(x, y, z, 0);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            wt.set(in, s2, ex, bc2 * w.cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nt) * Nex;
  exchange(size, &vt, &wt, 3, 1, 3);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * z);

        // boundary (t=Nt-1)
        int ix = idx.siteh(x, y, z, Nt - 1);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            v.set(in, ix, ex, vt.cmp(in, s2, ex));
          }
        }
      }
    }
  }
}


//====================================================================
void ShiftField_eo::dn_th(Field& v, const Field& w, const int bc,
                          const int ieo)
{
  double bc2;

  if (Communicator::ipe(3) == 0) {
    bc2 = bc;
  } else {
    bc2 = 1.0;
  }

  int Nin  = w.nin();
  int Nex  = w.nex();
  int Nvol = w.nvol();

  Field wt(Nin, Nvol / Nt, Nex);
  Field vt(Nin, Nvol / Nt, Nex);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * z);

        // bulk
        for (int t = 1; t < Nt; t++) {
          int ix = idx.siteh(x, y, z, t);
          int px = idx.siteh(x, y, z, t - 1);
          for (int ex = 0; ex < Nex; ex++) {
            for (int in = 0; in < Nin; in++) {
              v.set(in, ix, ex, w.cmp(in, px, ex));
            }
          }
        }
        // boundary (t=0)
        int ix = idx.siteh(x, y, z, 0);
        int px = idx.siteh(x, y, z, Nt - 1);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            wt.set(in, s2, ex, w.cmp(in, px, ex));
          }
        }
      }
    }
  }

  int size = Nin * (Nvol / Nt) * Nex;
  exchange(size, &vt, &wt, 3, -1, 7);

  for (int z = 0; z < Nz; z++) {
    for (int y = 0; y < Ny; y++) {
      for (int x = 0; x < Nx2; x++) {
        int s2 = x + Nx2 * (y + Ny * z);

        // boundary (t=0)
        int ix = idx.siteh(x, y, z, 0);
        int px = idx.siteh(x, y, z, Nt - 1);
        for (int ex = 0; ex < Nex; ex++) {
          for (int in = 0; in < Nin; in++) {
            v.set(in, ix, ex, bc2 * vt.cmp(in, s2, ex));
          }
        }
      }
    }
  }
}


//====================================================================
//============================================================END=====
