/*!
        @file    $Id:: fopr_CloverTerm_imp.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "fopr_CloverTerm.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;
using std::string;

//====================================================================
//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("hopping_parameter", 0.0);
    param.Register_double("clover_coefficient", 0.0);
    param.Register_int_vector("boundary_condition", std::valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Fopr_CloverTerm::Parameters_Fopr_CloverTerm() { append_entry(*this); }
//- end

//====================================================================
//! pimple prescription for Fopr_CloverTerm class implementation.
namespace {
  inline double mult_uv_r(double *u, double *v)
  {
    return u[0] * v[0] - u[1] * v[1]
           + u[2] * v[2] - u[3] * v[3]
           + u[4] * v[4] - u[5] * v[5];
  }


  inline double mult_uv_i(double *u, double *v)
  {
    return u[0] * v[1] + u[1] * v[0]
           + u[2] * v[3] + u[3] * v[2]
           + u[4] * v[5] + u[5] * v[4];
  }
}

//====================================================================
void Fopr_CloverTerm::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double        kappa, cSW;
  valarray<int> bc;

  int err = 0;
  err += params.fetch_double("hopping_parameter", kappa);
  err += params.fetch_double("clover_coefficient", cSW);
  err += params.fetch_int_vector("boundary_condition", bc);

  if (err) {
    vout.crucial(m_vl, "Fopr_CloverTerm: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(kappa, cSW, bc);
}


//====================================================================
void Fopr_CloverTerm::set_parameters(double kappa, double cSW,
                                     valarray<int> bc)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Fopr_CloverTerm:\n");
  vout.general(m_vl, "  kappa = %8.4f\n", kappa);
  vout.general(m_vl, "  cSW   = %8.4f\n", cSW);
  for (int mu = 0; mu < m_Ndim; ++mu) {
    vout.general(m_vl, "  boundary[%d] = %2d\n", mu, bc[mu]);
  }

  //- range check
  // NB. kappa,cSW == 0 is allowed.
  assert(bc.size() == m_Ndim);

  //- store values
  m_kappa = kappa;
  m_cSW   = cSW;

  // m_boundary.resize(m_Ndim);  // already resized in init.
  for (int mu = 0; mu < m_Ndim; ++mu) {
    m_boundary[mu] = bc[mu];
  }
}


//====================================================================
void Fopr_CloverTerm::set_config(Field *U)
{
  m_U = (Field_G *)U;
  set_csw();
}


//====================================================================
void Fopr_CloverTerm::init(string repr)
{
  m_Nvol = CommonParameters::Nvol();
  m_Ndim = CommonParameters::Ndim();
  m_Nc   = CommonParameters::Nc();
  m_Nd   = CommonParameters::Nd();
  m_NinF = 2 * m_Nc * m_Nd;

  m_U = 0;

  m_repr = repr;

  m_boundary.resize(m_Ndim);
  m_SG.resize(m_Ndim * m_Ndim);

  GammaMatrixSet *gmset = GammaMatrixSet::New(m_repr);

  m_GM5 = gmset->get_GM(gmset->GAMMA5);

  m_SG[sg_index(0, 1)] = gmset->get_GM(gmset->SIGMA12);
  m_SG[sg_index(1, 2)] = gmset->get_GM(gmset->SIGMA23);
  m_SG[sg_index(2, 0)] = gmset->get_GM(gmset->SIGMA31);
  m_SG[sg_index(3, 0)] = gmset->get_GM(gmset->SIGMA41);
  m_SG[sg_index(3, 1)] = gmset->get_GM(gmset->SIGMA42);
  m_SG[sg_index(3, 2)] = gmset->get_GM(gmset->SIGMA43);

  m_SG[sg_index(1, 0)] = m_SG[sg_index(0, 1)].mult(-1);
  m_SG[sg_index(2, 1)] = m_SG[sg_index(1, 2)].mult(-1);
  m_SG[sg_index(0, 2)] = m_SG[sg_index(2, 0)].mult(-1);
  m_SG[sg_index(0, 3)] = m_SG[sg_index(3, 0)].mult(-1);
  m_SG[sg_index(1, 3)] = m_SG[sg_index(3, 1)].mult(-1);
  m_SG[sg_index(2, 3)] = m_SG[sg_index(3, 2)].mult(-1);

  m_SG[sg_index(0, 0)] = gmset->get_GM(gmset->UNITY);
  m_SG[sg_index(1, 1)] = gmset->get_GM(gmset->UNITY);
  m_SG[sg_index(2, 2)] = gmset->get_GM(gmset->UNITY);
  m_SG[sg_index(3, 3)] = gmset->get_GM(gmset->UNITY);
  // these 4 gamma matrices are actually not used.


  //  m_fopr_w = new Fopr_Wilson(repr);
  if (m_repr == "Dirac") {
    m_csw = &Fopr_CloverTerm::mult_csw_dirac;
    m_gm5 = &Fopr_CloverTerm::gm5_dirac;
  } else if (m_repr == "Chiral") {
    m_csw = &Fopr_CloverTerm::mult_csw_chiral;
    m_gm5 = &Fopr_CloverTerm::gm5_chiral;
  }

  delete gmset;
}


//====================================================================
void Fopr_CloverTerm::tidyup()
{
  //  nothing to do.
}


//====================================================================
void Fopr_CloverTerm::mult_gm5(Field& v, const Field& f)
{
  (this->*m_gm5)(v, f);
}


//====================================================================
void Fopr_CloverTerm::gm5_dirac(Field& w, const Field& f)
{
  int Nvc = 2 * CommonParameters::Nc();
  int Nd  = CommonParameters::Nd();

  double *v1;
  double *v2;

  v1 = const_cast<Field *>(&f)->ptr(0);
  v2 = w.ptr(0);

  int id1 = 0;
  int id2 = Nvc;
  int id3 = Nvc * 2;
  int id4 = Nvc * 3;

  for (int site = 0; site < m_Nvol; ++site) {
    for (int icc = 0; icc < Nvc; icc++) {
      int in = Nvc * Nd * site;
      v2[icc + id1 + in] = v1[icc + id3 + in];
      v2[icc + id2 + in] = v1[icc + id4 + in];
      v2[icc + id3 + in] = v1[icc + id1 + in];
      v2[icc + id4 + in] = v1[icc + id2 + in];
    }
  }
}


//====================================================================
void Fopr_CloverTerm::gm5_chiral(Field& w, const Field& f)
{
  int Nvc = 2 * CommonParameters::Nc();
  int Nd  = CommonParameters::Nd();

  double *v1;
  double *v2;

  v1 = const_cast<Field *>(&f)->ptr(0);
  v2 = w.ptr(0);

  int id1 = 0;
  int id2 = Nvc;
  int id3 = Nvc * 2;
  int id4 = Nvc * 3;

  for (int site = 0; site < m_Nvol; ++site) {
    for (int icc = 0; icc < Nvc; icc++) {
      int in = Nvc * Nd * site;
      v2[icc + id1 + in] = v1[icc + id1 + in];
      v2[icc + id2 + in] = v1[icc + id2 + in];
      v2[icc + id3 + in] = -v1[icc + id3 + in];
      v2[icc + id4 + in] = -v1[icc + id4 + in];
    }
  }
}


//====================================================================
void Fopr_CloverTerm::mult_isigma(Field_F& v, const Field_F& w,
                                  const int mu, const int nu)
{
  assert(mu != nu);
  v.mult_iGM(m_SG[sg_index(mu, nu)], w);
}


//====================================================================
void Fopr_CloverTerm::mult_sigmaF(Field& v, const Field& f)
{
  mult_csw(v, f);
}


//====================================================================
void Fopr_CloverTerm::mult_csw(Field& v, const Field& w)
{
  (this->*m_csw)(v, w);
}


//====================================================================
void Fopr_CloverTerm::mult_csw_chiral(Field& v, const Field& w)
{
  assert(w.nex() == 1);

  int Nc   = CommonParameters::Nc();
  int Nvc  = 2 * Nc;
  int Ndf  = 2 * Nc * Nc;
  int Nd   = CommonParameters::Nd();
  int Nvol = w.nvol();

  int id1 = 0;
  int id2 = Nvc;
  int id3 = Nvc * 2;
  int id4 = Nvc * 3;

  double *w2 = const_cast<Field *>(&w)->ptr(0);
  double *v2 = v.ptr(0);

  double *Bx = m_Bx.ptr(0);
  double *By = m_By.ptr(0);
  double *Bz = m_Bz.ptr(0);
  double *Ex = m_Ex.ptr(0);
  double *Ey = m_Ey.ptr(0);
  double *Ez = m_Ez.ptr(0);

  v = 0.0;

  for (int site = 0; site < Nvol; ++site) {
    int iv = Nvc * Nd * site;
    int ig = Ndf * site;

    for (int ic = 0; ic < Nc; ++ic) {
      int icr = 2 * ic;
      int ici = icr + 1;
      int icg = ic * Nvc + ig;

      // isigma_23 * Bx
      v2[icr + id1 + iv] -= mult_uv_i(&Bx[icg], &w2[id2 + iv]);
      v2[ici + id1 + iv] += mult_uv_r(&Bx[icg], &w2[id2 + iv]);
      v2[icr + id2 + iv] -= mult_uv_i(&Bx[icg], &w2[id1 + iv]);
      v2[ici + id2 + iv] += mult_uv_r(&Bx[icg], &w2[id1 + iv]);

      v2[icr + id3 + iv] -= mult_uv_i(&Bx[icg], &w2[id4 + iv]);
      v2[ici + id3 + iv] += mult_uv_r(&Bx[icg], &w2[id4 + iv]);
      v2[icr + id4 + iv] -= mult_uv_i(&Bx[icg], &w2[id3 + iv]);
      v2[ici + id4 + iv] += mult_uv_r(&Bx[icg], &w2[id3 + iv]);

      // isigma_31 * By
      v2[icr + id1 + iv] += mult_uv_r(&By[icg], &w2[id2 + iv]);
      v2[ici + id1 + iv] += mult_uv_i(&By[icg], &w2[id2 + iv]);
      v2[icr + id2 + iv] -= mult_uv_r(&By[icg], &w2[id1 + iv]);
      v2[ici + id2 + iv] -= mult_uv_i(&By[icg], &w2[id1 + iv]);

      v2[icr + id3 + iv] += mult_uv_r(&By[icg], &w2[id4 + iv]);
      v2[ici + id3 + iv] += mult_uv_i(&By[icg], &w2[id4 + iv]);
      v2[icr + id4 + iv] -= mult_uv_r(&By[icg], &w2[id3 + iv]);
      v2[ici + id4 + iv] -= mult_uv_i(&By[icg], &w2[id3 + iv]);

      // isigma_12 * Bz
      v2[icr + id1 + iv] -= mult_uv_i(&Bz[icg], &w2[id1 + iv]);
      v2[ici + id1 + iv] += mult_uv_r(&Bz[icg], &w2[id1 + iv]);
      v2[icr + id2 + iv] += mult_uv_i(&Bz[icg], &w2[id2 + iv]);
      v2[ici + id2 + iv] -= mult_uv_r(&Bz[icg], &w2[id2 + iv]);

      v2[icr + id3 + iv] -= mult_uv_i(&Bz[icg], &w2[id3 + iv]);
      v2[ici + id3 + iv] += mult_uv_r(&Bz[icg], &w2[id3 + iv]);
      v2[icr + id4 + iv] += mult_uv_i(&Bz[icg], &w2[id4 + iv]);
      v2[ici + id4 + iv] -= mult_uv_r(&Bz[icg], &w2[id4 + iv]);

      // isigma_41 * Ex
      v2[icr + id1 + iv] += mult_uv_i(&Ex[icg], &w2[id2 + iv]);
      v2[ici + id1 + iv] -= mult_uv_r(&Ex[icg], &w2[id2 + iv]);
      v2[icr + id2 + iv] += mult_uv_i(&Ex[icg], &w2[id1 + iv]);
      v2[ici + id2 + iv] -= mult_uv_r(&Ex[icg], &w2[id1 + iv]);

      v2[icr + id3 + iv] -= mult_uv_i(&Ex[icg], &w2[id4 + iv]);
      v2[ici + id3 + iv] += mult_uv_r(&Ex[icg], &w2[id4 + iv]);
      v2[icr + id4 + iv] -= mult_uv_i(&Ex[icg], &w2[id3 + iv]);
      v2[ici + id4 + iv] += mult_uv_r(&Ex[icg], &w2[id3 + iv]);

      // isigma_42 * Ey
      v2[icr + id1 + iv] -= mult_uv_r(&Ey[icg], &w2[id2 + iv]);
      v2[ici + id1 + iv] -= mult_uv_i(&Ey[icg], &w2[id2 + iv]);
      v2[icr + id2 + iv] += mult_uv_r(&Ey[icg], &w2[id1 + iv]);
      v2[ici + id2 + iv] += mult_uv_i(&Ey[icg], &w2[id1 + iv]);

      v2[icr + id3 + iv] += mult_uv_r(&Ey[icg], &w2[id4 + iv]);
      v2[ici + id3 + iv] += mult_uv_i(&Ey[icg], &w2[id4 + iv]);
      v2[icr + id4 + iv] -= mult_uv_r(&Ey[icg], &w2[id3 + iv]);
      v2[ici + id4 + iv] -= mult_uv_i(&Ey[icg], &w2[id3 + iv]);

      // isigma_43 * Ez
      v2[icr + id1 + iv] += mult_uv_i(&Ez[icg], &w2[id1 + iv]);
      v2[ici + id1 + iv] -= mult_uv_r(&Ez[icg], &w2[id1 + iv]);
      v2[icr + id2 + iv] -= mult_uv_i(&Ez[icg], &w2[id2 + iv]);
      v2[ici + id2 + iv] += mult_uv_r(&Ez[icg], &w2[id2 + iv]);

      v2[icr + id3 + iv] -= mult_uv_i(&Ez[icg], &w2[id3 + iv]);
      v2[ici + id3 + iv] += mult_uv_r(&Ez[icg], &w2[id3 + iv]);
      v2[icr + id4 + iv] += mult_uv_i(&Ez[icg], &w2[id4 + iv]);
      v2[ici + id4 + iv] -= mult_uv_r(&Ez[icg], &w2[id4 + iv]);
    }
  }

  v *= m_kappa * m_cSW;
}


//====================================================================
void Fopr_CloverTerm::mult_csw_dirac(Field& v, const Field& w)
{
  assert(w.nex() == 1);

  int Nc   = CommonParameters::Nc();
  int Nvc  = 2 * Nc;
  int Ndf  = 2 * Nc * Nc;
  int Nd   = CommonParameters::Nd();
  int Nvol = w.nvol();

  int id1 = 0;
  int id2 = Nvc;
  int id3 = Nvc * 2;
  int id4 = Nvc * 3;

  double *w2 = const_cast<Field *>(&w)->ptr(0);
  double *v2 = v.ptr(0);

  double *Bx = m_Bx.ptr(0);
  double *By = m_By.ptr(0);
  double *Bz = m_Bz.ptr(0);
  double *Ex = m_Ex.ptr(0);
  double *Ey = m_Ey.ptr(0);
  double *Ez = m_Ez.ptr(0);

  v = 0.0;

  for (int site = 0; site < Nvol; ++site) {
    int iv = Nvc * Nd * site;
    int ig = Ndf * site;

    for (int ic = 0; ic < Nc; ++ic) {
      int icr = 2 * ic;
      int ici = icr + 1;
      int icg = ic * Nvc + ig;

      // isigma_23 * Bx
      v2[icr + id1 + iv] -= mult_uv_i(&Bx[icg], &w2[id2 + iv]);
      v2[ici + id1 + iv] += mult_uv_r(&Bx[icg], &w2[id2 + iv]);
      v2[icr + id2 + iv] -= mult_uv_i(&Bx[icg], &w2[id1 + iv]);
      v2[ici + id2 + iv] += mult_uv_r(&Bx[icg], &w2[id1 + iv]);

      v2[icr + id3 + iv] -= mult_uv_i(&Bx[icg], &w2[id4 + iv]);
      v2[ici + id3 + iv] += mult_uv_r(&Bx[icg], &w2[id4 + iv]);
      v2[icr + id4 + iv] -= mult_uv_i(&Bx[icg], &w2[id3 + iv]);
      v2[ici + id4 + iv] += mult_uv_r(&Bx[icg], &w2[id3 + iv]);

      // isigma_31 * By
      v2[icr + id1 + iv] += mult_uv_r(&By[icg], &w2[id2 + iv]);
      v2[ici + id1 + iv] += mult_uv_i(&By[icg], &w2[id2 + iv]);
      v2[icr + id2 + iv] -= mult_uv_r(&By[icg], &w2[id1 + iv]);
      v2[ici + id2 + iv] -= mult_uv_i(&By[icg], &w2[id1 + iv]);

      v2[icr + id3 + iv] += mult_uv_r(&By[icg], &w2[id4 + iv]);
      v2[ici + id3 + iv] += mult_uv_i(&By[icg], &w2[id4 + iv]);
      v2[icr + id4 + iv] -= mult_uv_r(&By[icg], &w2[id3 + iv]);
      v2[ici + id4 + iv] -= mult_uv_i(&By[icg], &w2[id3 + iv]);

      // isigma_12 * Bz
      v2[icr + id1 + iv] -= mult_uv_i(&Bz[icg], &w2[id1 + iv]);
      v2[ici + id1 + iv] += mult_uv_r(&Bz[icg], &w2[id1 + iv]);
      v2[icr + id2 + iv] += mult_uv_i(&Bz[icg], &w2[id2 + iv]);
      v2[ici + id2 + iv] -= mult_uv_r(&Bz[icg], &w2[id2 + iv]);

      v2[icr + id3 + iv] -= mult_uv_i(&Bz[icg], &w2[id3 + iv]);
      v2[ici + id3 + iv] += mult_uv_r(&Bz[icg], &w2[id3 + iv]);
      v2[icr + id4 + iv] += mult_uv_i(&Bz[icg], &w2[id4 + iv]);
      v2[ici + id4 + iv] -= mult_uv_r(&Bz[icg], &w2[id4 + iv]);

      // isigma_41 * Ex
      v2[icr + id1 + iv] += mult_uv_i(&Ex[icg], &w2[id4 + iv]);
      v2[ici + id1 + iv] -= mult_uv_r(&Ex[icg], &w2[id4 + iv]);
      v2[icr + id2 + iv] += mult_uv_i(&Ex[icg], &w2[id3 + iv]);
      v2[ici + id2 + iv] -= mult_uv_r(&Ex[icg], &w2[id3 + iv]);

      v2[icr + id3 + iv] += mult_uv_i(&Ex[icg], &w2[id2 + iv]);
      v2[ici + id3 + iv] -= mult_uv_r(&Ex[icg], &w2[id2 + iv]);
      v2[icr + id4 + iv] += mult_uv_i(&Ex[icg], &w2[id1 + iv]);
      v2[ici + id4 + iv] -= mult_uv_r(&Ex[icg], &w2[id1 + iv]);

      // isigma_42 * Ey
      v2[icr + id1 + iv] -= mult_uv_r(&Ey[icg], &w2[id4 + iv]);
      v2[ici + id1 + iv] -= mult_uv_i(&Ey[icg], &w2[id4 + iv]);
      v2[icr + id2 + iv] += mult_uv_r(&Ey[icg], &w2[id3 + iv]);
      v2[ici + id2 + iv] += mult_uv_i(&Ey[icg], &w2[id3 + iv]);

      v2[icr + id3 + iv] -= mult_uv_r(&Ey[icg], &w2[id2 + iv]);
      v2[ici + id3 + iv] -= mult_uv_i(&Ey[icg], &w2[id2 + iv]);
      v2[icr + id4 + iv] += mult_uv_r(&Ey[icg], &w2[id1 + iv]);
      v2[ici + id4 + iv] += mult_uv_i(&Ey[icg], &w2[id1 + iv]);

      // isigma_43 * Ez
      v2[icr + id1 + iv] += mult_uv_i(&Ez[icg], &w2[id3 + iv]);
      v2[ici + id1 + iv] -= mult_uv_r(&Ez[icg], &w2[id3 + iv]);
      v2[icr + id2 + iv] -= mult_uv_i(&Ez[icg], &w2[id4 + iv]);
      v2[ici + id2 + iv] += mult_uv_r(&Ez[icg], &w2[id4 + iv]);

      v2[icr + id3 + iv] += mult_uv_i(&Ez[icg], &w2[id1 + iv]);
      v2[ici + id3 + iv] -= mult_uv_r(&Ez[icg], &w2[id1 + iv]);
      v2[icr + id4 + iv] -= mult_uv_i(&Ez[icg], &w2[id2 + iv]);
      v2[ici + id4 + iv] += mult_uv_r(&Ez[icg], &w2[id2 + iv]);
    }
  }

  v *= m_kappa * m_cSW;
}


//====================================================================
void Fopr_CloverTerm::set_csw()
{
  set_fieldstrength(m_Bx, 1, 2);
  set_fieldstrength(m_By, 2, 0);
  set_fieldstrength(m_Bz, 0, 1);
  set_fieldstrength(m_Ex, 3, 0);
  set_fieldstrength(m_Ey, 3, 1);
  set_fieldstrength(m_Ez, 3, 2);
}


//====================================================================
void Fopr_CloverTerm::set_fieldstrength(Field_G& Fst,
                                        const int mu, const int nu)
{
  int Nvol = CommonParameters::Nvol();

  Staples staple;

  Field_G Cup(Nvol, 1), Cdn(Nvol, 1);
  Field_G Umu(Nvol, 1);
  Field_G v(Nvol, 1), v2(Nvol, 1);

  Cup = staple.upper(*m_U, mu, nu);
  Cdn = staple.lower(*m_U, mu, nu);
  Umu.setpart_ex(0, *m_U, mu);

  Fst.mult_Field_Gnd(0, Umu, 0, Cup, 0);
  Fst.multadd_Field_Gnd(0, Umu, 0, Cdn, 0, -1.0);

  v.mult_Field_Gdn(0, Cup, 0, Umu, 0);
  v.multadd_Field_Gdn(0, Cdn, 0, Umu, 0, -1.0);

  m_shift.forward(v2, v, mu);

  Fst += v2;

  Fst.ah_Field_G(0);
  Fst *= 0.25;
}


//====================================================================
//============================================================END=====
