/*!
        @file    $Id:: fopr_Wilson_impl.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef FOPR_WILSON_IMPL_INCLUDED
#define FOPR_WILSON_IMPL_INCLUDED

#include "fopr_Wilson.h"

#include "gammaMatrixSet.h"
#include "shiftField_lex.h"

#include "mat_SU_N.h"
#include "vec_SU_N.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Wilson fermion operator.

/*!
    This fermion operator defines the standard Wilson fermion.
    The gamma matrix representation is given as control string
    "Dirac"(default) or "Chiral" at the construction, which is
    used to construct the Fopr_Wilson instance.
    The `mode', which of D, Ddag, H, DdagD are multiplied, is
    controlled by setting the pointers to these functions,
    m_mult and m_mult_dag.
    At the beginning, they are set to point mult_undef() which
    just represent the mode has not been set.
    set_mode(string) must be called before mult() is called.
                                    [24 Dec 2011 H,Matsufuru]
 */

class Fopr_Wilson::Fopr_Wilson_impl
{
 public:

  Fopr_Wilson_impl() {}

  void init(std::string repr);

  void set_parameters(const double kappa, const std::valarray<int> bc);

  void set_config(Field *U)
  { m_U = (Field_G *)U; }

  void set_mode(std::string mode);

  std::string get_mode() const;

  inline void mult(Field& v, const Field& f)
  { (this->*m_mult)(v, f); }

  inline void mult_dag(Field& v, const Field& f)
  { (this->*m_mult_dag)(v, f); }

  inline void mult_gm5(Field& v, const Field& f)
  { (this->*m_gm5)(v, f); }

  inline void D(Field& v, const Field& f)
  { (this->*m_D)(v, f); }

  inline void Ddag(Field& w, const Field& f)
  {
    Field w2(f.nin(), f.nvol(), f.nex());

    mult_gm5(w, f);
    D(w2, w);
    mult_gm5(w, w2);
  }

  inline void DdagD(Field& w, const Field& f)
  {
    Field w2(f.nin(), f.nvol(), f.nex());

    D(w2, f);
    mult_gm5(w, w2);
    D(w2, w);
    mult_gm5(w, w2);
  }

  inline void H(Field& w, const Field& f)
  {
    Field w2(f.nin(), f.nvol(), f.nex());

    D(w2, f);
    mult_gm5(w, w2);
  }

  inline void mult_undef(Field&, const Field& f)
  {
    vout.crucial(m_vl, "Fopr_Wilson: mode undefined.\n");
    abort();
  }

  const Field_F mult_gm5p(int mu, const Field_F& w);

  void mult_up(int mu, Field& w, const Field& f);
  void mult_dn(int mu, Field& w, const Field& f);

  inline void fprop_normalize(Field& v)
  { v *= (2.0 * m_kappa); }

  inline void fopr_normalize(Field& v)
  { v *= 1.0 / (2.0 * m_kappa); }


 private:

  // prohibit copy
  Fopr_Wilson_impl(const Fopr_Wilson::Fopr_Wilson_impl&) {}
  Fopr_Wilson_impl& operator=(const Fopr_Wilson::Fopr_Wilson_impl&);


  Bridge::VerboseLevel m_vl;

  int m_Nvol;
  int m_Ndim;

  double             m_kappa;
  std::valarray<int> m_boundary;

  std::string m_mode;
  std::string m_repr;

  void (Fopr_Wilson::Fopr_Wilson_impl::*m_mult) (Field&, const Field&);
  void (Fopr_Wilson::Fopr_Wilson_impl::*m_mult_dag) (Field&, const Field&);
  void (Fopr_Wilson::Fopr_Wilson_impl::*m_D) (Field&, const Field&);
  void (Fopr_Wilson::Fopr_Wilson_impl::*m_gm5) (Field&, const Field&);
  void (Fopr_Wilson::Fopr_Wilson_impl::*m_mult_tp) (Field&, const Field&);
  void (Fopr_Wilson::Fopr_Wilson_impl::*m_mult_tm) (Field&, const Field&);

  const Field_G *m_U;

  std::valarray<GammaMatrix> m_GM;

//  ShiftField_lex shift;
//  Field_F trf, trf2;

  void D_chiral(Field&, const Field&);
  void D_dirac(Field&, const Field&);
  void gm5_chiral(Field&, const Field&);
  void gm5_dirac(Field&, const Field&);

  void mult_p(int mu, Field_F&, const Field_F&);
  void mult_m(int mu, Field_F&, const Field_F&);

  void mult_xp(Field&, const Field&);
  void mult_xm(Field&, const Field&);
  void mult_yp(Field&, const Field&);
  void mult_ym(Field&, const Field&);
  void mult_zp(Field&, const Field&);
  void mult_zm(Field&, const Field&);

  void mult_tp_dirac(Field&, const Field&);
  void mult_tm_dirac(Field&, const Field&);
  void mult_tp_chiral(Field&, const Field&);
  void mult_tm_chiral(Field&, const Field&);
};
#endif
