/*!
        @file    $Id:: fopr.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/


#ifndef FOPR_INCLUDED
#define FOPR_INCLUDED

#include "parameters.h"
#include "commonParameters.h"
#include "communicator.h"
#include "field.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Base class of fermion operator family.

/*!
    In Bridge-OF, the fermion operator implies an operator which
    transforms a field to other field, irrespective of physical
    formulation.
    This class defines the interface of the fermion operators.
    At present, void functions mult(v,w) and mult_dag(v,w) is
    not purely virtual, because some of subclass have not
    implemented them yet.
                                     [20 Dec 2011 H.Matsufuru]
*/

class Fopr
{
 public:

  Fopr()
    : m_vl(CommonParameters::Vlevel()) {}

  virtual ~Fopr() {}

  virtual void set_parameters(const Parameters&) = 0;

  void set_parameter_verboselevel(const Bridge::VerboseLevel vl) { m_vl = vl; }

  //! setting pointer to the gauge configuration.
  virtual void set_config(Field *) = 0;

  //! \brief multiplies fermion operator to a given field and returns
  //!  the resultant field.
  virtual const Field mult(const Field&) = 0;

  //! hermitian conjugate of mult(const Field&).
  virtual const Field mult_dag(const Field&) { return Field(); }

  //! \brief multiplies fermion operator to a given field (2nd argument)
  //   and set the resultant field to the 1st argument.
  virtual void mult(Field&, const Field&) {}

  //! hermitian conjugate of mult(Field&, const Field&).
  virtual void mult_dag(Field&, const Field&) {}

  //! nearest neighbor hopping term: temporary entry [H.Matsufuru]
  virtual void mult_up(int mu, Field&, const Field&) {}
  virtual void mult_dn(int mu, Field&, const Field&) {}

  //! \brief setting the mode of multiplication if necessary.
  //!  Default implementation here is just to avoid irrelevant call.
  virtual void set_mode(std::string mode)
  {
    vout.general(m_vl, "Fopr: set_mode not implemented.\n");
  }

  virtual std::string get_mode() const
  {
    vout.general(m_vl, "Fopr: get_mode not implemented.\n");
    return std::string();
  }

  //! returns the volume for which the fermion operator is defined.
  virtual int field_nvol() = 0;

  //! returns the on-site d.o.f. for which the fermion operator is defined.
  virtual int field_nin() = 0;

  //! returns the external d.o.f. for which the fermion operator is defined.
  virtual int field_nex() = 0;

 protected:
  Bridge::VerboseLevel m_vl;
};
#endif
