/*!
        @file    $Id:: fopr_Clover.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "fopr_Clover.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;
using std::string;

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_string("gamma_matrix_type", "NULL");

    param.Register_double("hopping_parameter", 0.0);
    param.Register_double("clover_coefficient", 0.0);
    param.Register_int_vector("boundary_condition", std::valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Fopr.Clover", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Fopr_Clover::Parameters_Fopr_Clover() { append_entry(*this); }
//- end

//====================================================================
void Fopr_Clover::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double        kappa, cSW;
  valarray<int> bc;

  int err = 0;
  err += params.fetch_double("hopping_parameter", kappa);
  err += params.fetch_double("clover_coefficient", cSW);
  err += params.fetch_int_vector("boundary_condition", bc);

  if (err) {
    vout.crucial(m_vl, "Fopr_Clover: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(kappa, cSW, bc);
}


//====================================================================
void Fopr_Clover::set_parameters(double kappa, double cSW, valarray<int> bc)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of clover fermion operator:\n");
  vout.general(m_vl, "  kappa = %8.4f\n", kappa);
  vout.general(m_vl, "  cSW   = %8.4f\n", cSW);
  for (int mu = 0; mu < m_Ndim; ++mu) {
    vout.general(m_vl, "  boundary[%d] = %2d\n", mu, bc[mu]);
  }

  //- range check
  // NB. kappa,cSW == 0 is allowed.
  assert(bc.size() == m_Ndim);

  //- store values
  m_kappa = kappa;
  m_cSW   = cSW;

  for (int mu = 0; mu < m_Ndim; ++mu) {
    m_boundary[mu] = bc[mu];
  }

  //- propagate parameters to components
  m_fopr_w->set_parameters(m_kappa, m_boundary);
  m_fopr_csw->set_parameters(m_kappa, m_cSW, m_boundary);
}


//====================================================================
void Fopr_Clover::init(string repr)
{
  m_Nvol = CommonParameters::Nvol();
  m_Ndim = CommonParameters::Ndim();
  m_Nc   = CommonParameters::Nc();
  m_Nd   = CommonParameters::Nd();
  m_NinF = 2 * m_Nc * m_Nd;

  m_boundary.resize(m_Ndim);

  m_U = 0;

  m_repr = repr;

  m_fopr_w   = new Fopr_Wilson(repr);
  m_fopr_csw = new Fopr_CloverTerm(repr);
}


//====================================================================
void Fopr_Clover::tidyup()
{
  delete m_fopr_w;
  delete m_fopr_csw;
}


//====================================================================
void Fopr_Clover::DdagD(Field& w, const Field& f)
{
  assert(f.nex() == 1);
  Field w2(f.nin(), f.nvol(), 1);

  D(w2, f);
  mult_gm5(w, w2);
  D(w2, w);
  mult_gm5(w, w2);
}


//====================================================================
void Fopr_Clover::Ddag(Field& w, const Field& f)
{
  assert(f.nex() == 1);
  Field w2(f.nin(), f.nvol(), 1);

  mult_gm5(w, f);
  D(w2, w);
  mult_gm5(w, w2);
}


//====================================================================
void Fopr_Clover::H(Field& w, const Field& f)
{
  assert(f.nex() == 1);
  Field w2(f.nin(), f.nvol(), 1);

  D(w2, f);
  mult_gm5(w, w2);
}


//====================================================================
void Fopr_Clover::D(Field& w, const Field& f)
{
  assert(f.nex() == 1);
  Field w2(f.nin(), f.nvol(), 1);

  m_fopr_w->D(w, f);
  m_fopr_csw->mult_sigmaF(w2, f);
  w -= w2;
}


//====================================================================
void Fopr_Clover::mult_isigma(Field_F& v, const Field_F& w,
                              const int mu, const int nu)
{
  m_fopr_csw->mult_isigma(v, w, mu, nu);
}


//====================================================================
//============================================================END=====
