/*!
        @file    $Id:: fopr_Clover.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef FOPR_CLOVER_INCLUDED
#define FOPR_CLOVER_INCLUDED

#include "fopr_Wilson.h"
#include "fopr_CloverTerm.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Clover fermion operator.

/*!
    This class implements the clover (improved Wilson) fermion
    operator.
    Wilson kernel and clover term are implemented in other classes,
    and this class holds them as objects.
    (The implementation was modified after revision 645: before
     that, clover term was being implemented inside this class.)
    The `mode' controls which of D, Ddag, H, DdagD are multiplied
    when mult or mult_dag is called.
       [first ver. 24 Dec 2011/ modified 28 Aug 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.             [14 Nov 2012 Y.Namekawa]
 */

//- parameters class
class Parameters_Fopr_Clover : virtual public Parameters
{
 public:
  Parameters_Fopr_Clover();
};
//- end

class Fopr_Clover : public Fopr
{
 private:
  double             m_kappa;             //!< hopping parameter
  double             m_cSW;               //!< clover coefficient
  std::valarray<int> m_boundary;          //!< boundary conditions
  std::string        m_repr;              //!<  gamma matrix representation
  std::string        m_mode;              //!<  mode of multiplication

  int m_Nvol, m_Ndim, m_Nc, m_Nd, m_NinF; //!< internal parameters

  Fopr_Wilson     *m_fopr_w;              //!< Wilson fermion kernel
  Fopr_CloverTerm *m_fopr_csw;            //!< Clover term operator
  const Field_G   *m_U;                   //!< gauge configuration (pointer)

  //  Fopr_Clover_imp* m_imp;  //!< pimple prescription

 public:

  Fopr_Clover()
    : Fopr()
  {
    init("Dirac");
  }

  Fopr_Clover(std::string repr)
    : Fopr()
  {
    init(repr);
  }

  ~Fopr_Clover()
  {
    tidyup();
  }

  void set_parameters(const Parameters& params);
  void set_parameters(double kappa, double cSW, std::valarray<int> bc);

  void set_config(Field *U)
  {
    m_U = (Field_G *)U;
    m_fopr_w->set_config(U);
    m_fopr_csw->set_config(U);
  }

  void set_mode(std::string mode)
  {
    m_mode = mode;
  }

  std::string get_mode() const
  {
    return m_mode;
  }

  const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  void mult(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      D(v, f);
    } else if (m_mode == "DdagD") {
      DdagD(v, f);
    } else if (m_mode == "Ddag") {
      Ddag(v, f);
    } else if (m_mode == "H") {
      H(v, f);
    } else {
      vout.crucial(m_vl, "Fopr_Clover: undefined mode = %s.\n", m_mode.c_str());
      abort();
    }
  }

  void mult_dag(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      Ddag(v, f);
    } else if (m_mode == "DdagD") {
      DdagD(v, f);
    } else if (m_mode == "Ddag") {
      D(v, f);
    } else if (m_mode == "H") {
      H(v, f);
    } else {
      vout.crucial(m_vl, "Fopr_Clover: undefined mode = %s.\n", m_mode.c_str());
      abort();
    }
  }

  void DdagD(Field&, const Field&);
  void D(Field&, const Field&);
  void Ddag(Field&, const Field&);
  void H(Field&, const Field&);

  void mult_gm5(Field& v, const Field& w)
  {
    m_fopr_w->mult_gm5(v, w);
  }

  const Field mult_gm5(const Field& w)
  {
    return m_fopr_w->mult_gm5(w);
  }

  void mult_up(int mu, Field& v, const Field& w)
  {
    m_fopr_w->mult_up(mu, v, w);
  }

  void mult_dn(int mu, Field& v, const Field& w)
  {
    m_fopr_w->mult_dn(mu, v, w);
  }

  void mult_isigma(Field_F&, const Field_F&,
                   const int mu, const int nu);

  int field_nvol() { return m_Nvol; }
  int field_nin() { return 2 * m_Nc * m_Nd; }
  int field_nex() { return 1; }

 private:

  void init(std::string repr);
  void tidyup();
};
#endif
