/*!
        @file    $Id:: fopr_CloverTerm.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef FOPR_CLOVERTERM_INCLUDED
#define FOPR_CLOVERTERM_INCLUDED

#include "fopr_Wilson.h"
#include "shiftField_lex.h"
#include "gammaMatrixSet.h"
#include "staples.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Clover term operator.

/*!
    This class implements the clover term for the clover (improved
    Wilson) fermion operator.
    This part was separated from the Fopr_Clover class.
    The field strength is calculate when the function
    set_config() is called.
    The `mode' for setting fermion operator mode is now only
    defined to the case 'D'.
                [30 Sep 2012 H.Matsufuru,
                 original clover operator: 24 Dec 2011 H.M.]
    (Coding history will be recovered from trac.)
    YAML is implemented.            [14 Nov 2012 Y.Namekawa]
    Selector is implemented.        [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

//- parameters class
class Parameters_Fopr_CloverTerm : virtual public Parameters
{
 public:
  Parameters_Fopr_CloverTerm();
};
//- end

class Fopr_CloverTerm : public Fopr
{
 private:
  double             m_kappa;
  double             m_cSW;
  std::valarray<int> m_boundary;
  std::string        m_repr;
  std::string        m_mode;
  void               (Fopr_CloverTerm::*m_csw) (Field&, const Field&);
  void               (Fopr_CloverTerm::*m_gm5) (Field&, const Field&);

  int m_Nvol, m_Ndim, m_Nc, m_Nd, m_NinF;

  const Field_G  *m_U;
  ShiftField_lex m_shift;

  Field_G m_Bx, m_By, m_Bz, m_Ex, m_Ey, m_Ez;
  // Bx = -iF(1,2), By = -iF(2,1), -iBz = F(0,1)
  // Ex = -iF(4,0), Ey = -iF(4,1), Ez = -iF(4,2)

  std::valarray<GammaMatrix> m_SG;
  GammaMatrix                m_GM5;

 public:
  Fopr_CloverTerm()
    : Fopr()
  {
    init("Dirac");
  }

  Fopr_CloverTerm(std::string repr)
    : Fopr()
  {
    init(repr);
  }

  ~Fopr_CloverTerm()
  {
    tidyup();
  }

  void set_parameters(const Parameters& params);
  void set_parameters(double kappa, double cSW, std::valarray<int> bc);
  void set_config(Field *U);

  void set_mode(std::string mode)
  {
    m_mode = mode;
  }

  std::string get_mode() const
  {
    return m_mode;
  }

  const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  void mult(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      mult_sigmaF(v, f);
//  } else if(m_mode=="H"){
//    H(v,f);
    } else {
      vout.crucial("Fopr_CloverTerm: undefined mode = %s\n", m_mode.c_str());
      abort();
    }
  }

  void mult_dag(Field& v, const Field& f)
  {
    mult(v, f);
  }

  void mult_sigmaF(Field&, const Field&);

  const Field mult_gm5(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_gm5(v, f);
    return v;
  }

  void mult_gm5(Field& v, const Field& w);

  void mult_isigma(Field_F&, const Field_F&,
                   const int mu, const int nu);

  int field_nvol() { return m_Nvol; }
  int field_nin() { return 2 * m_Nc * m_Nd; }
  int field_nex() { return 1; }

 private:

  void init(std::string repr);
  void tidyup();

  void set_csw();
  void set_fieldstrength(Field_G&, const int, const int);

  void gm5_dirac(Field&, const Field&);
  void gm5_chiral(Field&, const Field&);

  void mult_csw(Field&, const Field&);
  void mult_csw_dirac(Field&, const Field&);
  void mult_csw_chiral(Field&, const Field&);

  void mult_csw_dirac(Field_F&, const Field_F&);
  void mult_csw_chiral(Field_F&, const Field_F&);

  int sg_index(int mu, int nu) { return mu * m_Ndim + nu; }
};
#endif
