/*!
        @file    $Id:: fopr_CloverTerm_eo.h #$

        @brief

        @author  <UEDA, Satoru> sueda@post.kek.jp(sueda)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef FOPR_CLOVERTERM_EO_INCLUDED
#define FOPR_CLOVERTERM_EO_INCLUDED

#include <vector>

#include "fopr_Wilson_eo.h"
#include "staples_eo.h"

#include "gammaMatrixSet.h"

#include "mat_SU_N.h"
#include "vec_SU_N.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Clover term operator.

/*!
    This class implements the clover term for the clover (improved
    Wilson) fermion operator.
    This part was separated from the Fopr_Clover class.
    The field strength is calculate when the function
    set_config() is called.
    The `mode' for setting fermion operator mode is now only
    defined to the case 'D'.
                [30 Sep 2012 H.Matsufuru,
                 original clover operator: 24 Dec 2011 H.M.]
    (Coding history will be recovered from trac.)
    Modify this code to work.      [03 Mar 2013 Y.Namekawa]
 */

//- parameters class
class Parameters_Fopr_CloverTerm_eo : virtual public Parameters
{
 public:
  Parameters_Fopr_CloverTerm_eo();
};
//- end

class Fopr_CloverTerm_eo : public Fopr
{
// This class return D_ee = 1-f_ee or D_oo = 1-f_oo
 private:
  int                m_Nvol, m_Nvol2, m_Ndim, m_Nc, m_Nd, m_NinF, m_Ndm2;
  double             m_kappa, m_cSW;
  std::valarray<int> m_boundary;
  std::string        m_repr;
  std::string        m_mode;

  //! Gamma Matrix and Sigma_{mu,nu} = -i [Gamma_mu, Gamma_nu] /2
  std::valarray<GammaMatrix> m_GM, m_SG;

  void (Fopr_CloverTerm_eo::*m_mult) (Field&, const Field&);

  Index_eo idx;

  const Field_G *m_Ueo;
  ShiftField_eo m_shift_eo;

  //! m_T = 1 - kappa c_SW sigma F / 2
  Field_G m_T;

  Vec_SU_N v1, v2;

 public:
  Fopr_CloverTerm_eo(std::string repr)
    : m_Nvol(CommonParameters::Nvol()),
      m_Nvol2(m_Nvol / 2),
      m_Ndim(CommonParameters::Ndim()),
      m_Nc(CommonParameters::Nc()),
      m_Nd(CommonParameters::Nd()),
      m_Ndm2(m_Nd * m_Nd / 2),
      m_T(m_Nvol, m_Ndm2)
  {
    init(repr);
  }

  ~Fopr_CloverTerm_eo() {}

  void set_parameters(const Parameters& params);
  void set_parameters(const double kappa, const double cSW,
                      const std::valarray<int> bc);
  void set_config(Field *Ueo);

  void set_mode(std::string mode)
  {
    m_mode = mode;
  }

  std::string get_mode() const
  {
    return m_mode;
  }

  //! return D = D^dag = 1-f_ee or 1-f_oo
  const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  void mult(Field& v, const Field& f)
  {
    if (m_mode == "even") {
      D(v, f, 0);
    } else if (m_mode == "odd") {
      D(v, f, 1);
    } else {
      vout.crucial("Fopr_CloverTerm_eo: undefined mode = %s\n", m_mode.c_str());
      abort();
    }
  }

  void mult_dag(Field& v, const Field& f)
  {
    mult(v, f);
  }

  void mult_isigma(Field_F&, const Field_F&,
                   const int mu, const int nu);

  const Field D(const Field& f, const int ieo)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    D(v, f, ieo);
    return v;
  }

  void D(Field& v, const Field& f, const int ieo);

  std::vector<double> csmatrix(const int&);

  int field_nvol() { return m_Nvol2; }
  int field_nin() { return 2 * m_Nc * m_Nd; }
  int field_nex() { return 1; }

 private:
  void init(std::string repr);

  void set_csw();
  void mult_csw(Field_F&, const Field_F&, const int ieo);
  void set_fieldstrength(Field_G&, const int, const int);

  int sg_index(int mu, int nu) { return mu * m_Ndim + nu; }
};
#endif
