/*!
        @file    $Id:: fopr_Clover_SF.h #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef FOPR_CLOVER_SF_INCLUDED
#define FOPR_CLOVER_SF_INCLUDED

#include "fopr_Wilson_SF.h"
#include "staples_SF.h"

#include "shiftField_lex.h"
#include "gammaMatrixSet.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Clover fermion operator.

/*!
    This class implements the clover (improved Wilson) fermion
    operator with SF BC.
    <ul>
    <li>The field strength is calculate when the function set_config() is called.
    <li>Dirac representation only!
    <li>[10 Apr 2012 Y.Taniguchi]
    </ul>
    (Coding history will be recovered from trac.)
    YAML is implemented.             [14 Nov 2012 Y.Namekawa]
 */

//- parameters class
class Parameters_Fopr_Clover_SF : virtual public Parameters
{
 public:
  Parameters_Fopr_Clover_SF();
};
//- end

class Fopr_Clover_SF : public Fopr
{
 private:
  int                m_Nvol, m_Ndim, m_Nc, m_Nd, m_NinF;
  double             m_kappa, m_cSW;
  std::valarray<int> m_boundary;
  std::string        m_repr;
  std::string        m_mode;

  void (Fopr_Clover_SF::*m_csw) (Field_F&, const Field_F&);

  //  Index_lex m_idx;
  Fopr_Wilson_SF *m_fopr_w;
  const Field_G  *m_U;
  ShiftField_lex m_shift;

  Field_G m_Bx, m_By, m_Bz, m_Ex, m_Ey, m_Ez;
  // Bx = -iF(1,2), By = -iF(2,1), -iBz = F(0,1)
  // Ex = -iF(4,0), Ey = -iF(4,1), Ez = -iF(4,2)

  std::valarray<GammaMatrix> m_GM, m_SG;

  //! SF boundary condition at t=0
  double m_phi[3];
  //! SF boundary condition at t=Nt
  double m_phipr[3];

  //! In order to set the boundary field to zero.
  Field_F_SF setzero;

 public:
  Fopr_Clover_SF()
  {
    init("Dirac");
  }

  /*
  Fopr_Clover_SF(std::string repr){
    init(repr);
  }
  */

  ~Fopr_Clover_SF()
  {
    tidyup();
  }

  void set_parameters(const Parameters& params);
  void set_parameters(double kappa, double cSW, std::valarray<int> bc,
                      double *phi, double *phipr);

  void set_config(Field *U)
  {
    m_U = (Field_G *)U;
    m_fopr_w->set_config(U);

    set_csw();
  }

  void set_mode(std::string mode)
  {
    m_mode = mode;
  }

  std::string get_mode() const
  {
    return m_mode;
  }

  const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  void mult(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      D(v, f);
    } else if (m_mode == "DdagD") {
      DdagD(v, f);
    } else if (m_mode == "Ddag") {
      Ddag(v, f);
    } else if (m_mode == "H") {
      H(v, f);
    } else {
      vout.crucial(m_vl, "Fopr_Clover_SF: undefined mode = %s.\n", m_mode.c_str());
      abort();
    }
  }

  void mult_dag(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      Ddag(v, f);
    } else if (m_mode == "DdagD") {
      DdagD(v, f);
    } else if (m_mode == "Ddag") {
      D(v, f);
    } else if (m_mode == "H") {
      H(v, f);
    } else {
      vout.crucial(m_vl, "Fopr_Clover_SF: undefined mode = %s.\n", m_mode.c_str());
      abort();
    }
  }

  const Field_F DdagD(const Field_F&);
  const Field_F D(const Field_F&);
  const Field_F Ddag(const Field_F&);
  const Field_F H(const Field_F&);

  const Field mult_gm5(const Field& w)
  {
    return m_fopr_w->mult_gm5(w);
  }

  void DdagD(Field&, const Field&);
  void D(Field&, const Field&);
  void Ddag(Field&, const Field&);
  void H(Field&, const Field&);

  void mult_gm5(Field& v, const Field& w)
  {
    m_fopr_w->mult_gm5(v, w);
  }

  void mult_isigma(Field_F&, const Field_F&,
                   const int mu, const int nu);

  int field_nvol() { return m_Nvol; }
  int field_nin() { return 2 * m_Nc * m_Nd; }
  int field_nex() { return 1; }

 private:
  void init(std::string repr);
  void tidyup();

  void set_csw();
  void mult_csw(Field_F&, const Field_F&);
  void set_fieldstrength(Field_G&, const int, const int);

  void mult_csw_dirac(Field_F&, const Field_F&);
  void mult_csw_chiral(Field_F&, const Field_F&);

  int sg_index(int mu, int nu) { return mu * m_Ndim + nu; }
};
#endif
