/*!
        @file    $Id:: fopr_Rational.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "fopr_Rational.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

//- parameter entry
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_int("number_of_poles", 0);
    param.Register_int("exponent_numerator", 0);
    param.Register_int("exponent_denominator", 0);
    param.Register_double("lower_bound", 0.0);
    param.Register_double("upper_bound", 0.0);
    param.Register_int("maximum_number_of_iteration", 0);
    param.Register_double("convergence_criterion_squared", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Fopr.Rational", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Fopr_Rational::Parameters_Fopr_Rational() { append_entry(*this); }
//- end

//====================================================================
void Fopr_Rational::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  int    Np, n_exp, d_exp;
  double x_min, x_max;
  int    Niter;
  double Stop_cond;

  int err = 0;
  err += params.fetch_int("number_of_poles", Np);
  err += params.fetch_int("exponent_numerator", n_exp);
  err += params.fetch_int("exponent_denominator", d_exp);
  err += params.fetch_double("lower_bound", x_min);
  err += params.fetch_double("upper_bound", x_max);
  err += params.fetch_int("maximum_number_of_iteration", Niter);
  err += params.fetch_double("convergence_criterion_squared", Stop_cond);

  if (err) {
    vout.crucial(m_vl, "Fopr_Rational: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(Np, n_exp, d_exp, x_min, x_max, Niter, Stop_cond);
}


//====================================================================
void Fopr_Rational::set_parameters(int Np, int n_exp, int d_exp,
                                   double x_min, double x_max,
                                   int Niter, double Stop_cond)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Fopr_Rational:\n");
  vout.general(m_vl, "  Np        = %d\n", Np);
  vout.general(m_vl, "  n_exp     = %d\n", n_exp);
  vout.general(m_vl, "  d_exp     = %d\n", d_exp);
  vout.general(m_vl, "  x_min     = %10.6f\n", x_min);
  vout.general(m_vl, "  x_max     = %10.6f\n", x_max);
  vout.general(m_vl, "  Niter     = %d\n", Niter);
  vout.general(m_vl, "  Stop_cond = %10.6f\n", Stop_cond);

  //- range check
  int err = 0;
  err += ParameterCheck::non_zero(Np);
  err += ParameterCheck::non_zero(n_exp);
  err += ParameterCheck::non_zero(d_exp);
  // NB. x_min,x_max=0 is allowed.
  err += ParameterCheck::non_zero(Niter);
  err += ParameterCheck::square_non_zero(Stop_cond);

  if (err) {
    vout.crucial(m_vl, "Fopr_Rational: parameter range check failed.\n");
    abort();
  }

  //- store values
  m_Np        = Np;
  m_n_exp     = n_exp;
  m_d_exp     = d_exp;
  m_x_min     = x_min;
  m_x_max     = x_max;
  m_Niter     = Niter;
  m_Stop_cond = Stop_cond;

  //- post-process
  init_parameters();
}


//====================================================================
void Fopr_Rational::init_parameters()
{
  int Nin  = m_fopr->field_nin();
  int Nvol = m_fopr->field_nvol();
  int Nex  = m_fopr->field_nex();

  int Nshift = m_Np;

  double x_min2 = m_x_min * m_x_min;
  double x_max2 = m_x_max * m_x_max;

  m_cl.resize(m_Np);
  m_bl.resize(m_Np);

  m_xq.resize(m_Np);
  for (int i = 0; i < Nshift; ++i) {
    m_xq[i].reset(Nin, Nvol, Nex);
  }

  m_solver = new Shiftsolver_CG(m_fopr, m_Niter, m_Stop_cond);

  Math_Rational rational;
  rational.set_parameters(m_Np, m_n_exp, m_d_exp, x_min2, x_max2);
  rational.get_parameters(m_a0, m_bl, m_cl);

  vout.general(m_vl, " a0     = %18.14f\n", m_a0);
  for (int i = 0; i < m_Np; i++) {
    vout.general(m_vl, " bl[%d] = %18.14f  cl[%d] = %18.14f\n",
                 i, m_bl[i], i, m_cl[i]);
  }
}


//====================================================================
const Field Fopr_Rational::mult(const Field& b)
{
  int    Nconv;
  double diff;

  vout.general(m_vl, "    Shift solver in rational function\n");
  vout.general(m_vl, "      Number of shift values = %d\n", m_Np);

  m_fopr->set_mode("DdagD");
  m_solver->solve(m_xq, m_cl, b, Nconv, diff);

  vout.general(m_vl, "      diff(max) = %22.15e  \n", diff);

  Field v(b);
  v *= m_a0;
  for (int i = 0; i < m_Np; i++) {
    v += m_bl[i] * m_xq[i];
  }

  return v;
}


//====================================================================
double Fopr_Rational::func(double x)
{
  double y = m_a0;

  for (int k = 0; k < m_Np; ++k) {
    y += m_bl[k] / (x + m_cl[k]);
  }

  return y;
}


//====================================================================
//============================================================END=====
