/*!
        @file    $Id:: fopr_Wilson.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef FOPR_WILSON_INCLUDED
#define FOPR_WILSON_INCLUDED

#include <valarray>
#include <string>
#include <cassert>

#include "fopr.h"
#include "field_F.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Wilson fermion operator.

/*!
    This fermion operator defines the standard Wilson fermion.
    The gamma matrix representation is given as control string
    "Dirac"(default) or "Chiral" at the construction, which is
    used to construct the Fopr_Wilson instance.
    The `mode', which of D, Ddag, H, DdagD are multiplied, is
    controlled by setting the pointers to these functions,
    m_mult and m_mult_dag.
    At the beginning, they are set to point mult_undef() which
    just represent the mode has not been set.
    set_mode(string) must be called before mult() is called.
                                    [24 Dec 2011 H,Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.            [14 Nov 2012 Y.Namekawa]
 */

//- parameters class
class Parameters_Fopr_Wilson : virtual public Parameters
{
 public:
  Parameters_Fopr_Wilson();
};
//- end

class Fopr_Wilson : public Fopr
{
 public:
  // interface definitions

  Fopr_Wilson();
  Fopr_Wilson(std::string repr);

  ~Fopr_Wilson();

  void set_parameters(const Parameters& params);
  void set_parameters(const double kappa, const std::valarray<int> bc);

  void set_config(Field *U);

  void set_mode(std::string mode);

  std::string get_mode() const;

  inline const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  inline const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  inline const Field mult_gm5(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_gm5(v, f);
    return v;
  }

  void mult(Field& v, const Field& f);
  void mult_dag(Field& v, const Field& f);
  void mult_gm5(Field& w, const Field& v);
  void D(Field& w, const Field& v);
  void Ddag(Field& w, const Field& v);
  void DdagD(Field& w, const Field& v);
  void H(Field& w, const Field& v);

  const Field_F mult_gm5p(int mu, const Field_F& w);

  //! adding the hopping to nearest neighbor site in mu-th direction.
  void mult_up(int mu, Field& w, const Field& v);
  void mult_dn(int mu, Field& w, const Field& v);

  void fprop_normalize(Field& v);
  void fopr_normalize(Field& v);

  int field_nvol()
  { return CommonParameters::Nvol(); }
  int field_nin()
  { return 2 * CommonParameters::Nc() * CommonParameters::Nd(); }
  int field_nex()
  { return 1; }

 private:
  // pimple prescription for Fopr_Wilson class implementation.
  class Fopr_Wilson_impl;

  Fopr_Wilson_impl *m_impl;
};
#endif
