/*!
        @file    $Id:: fopr_Wilson_SF.cpp #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "fopr_Wilson_SF.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("hopping_parameter", 0.0);
    param.Register_int_vector("boundary_condition", std::valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Fopr.Wilson_SF", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Fopr_Wilson_SF::Parameters_Fopr_Wilson_SF() { append_entry(*this); }
//- end

//====================================================================
void Fopr_Wilson_SF::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double        kappa;
  valarray<int> bc;

  int err = 0;
  err += params.fetch_double("hopping_parameter", kappa);
  err += params.fetch_int_vector("boundary_condition", bc);

  if (err) {
    vout.crucial(m_vl, "Fopr_Wilson_SF: fetch error, input parameter not found.\n");
    abort();
  }

  set_parameters(kappa, bc);
}


//====================================================================
void Fopr_Wilson_SF::set_parameters(const double kappa, const std::valarray<int> bc)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Fopr_Wilson_SF:\n");
  vout.general(m_vl, "  kappa = %8.4f\n", kappa);
  for (int dir = 0; dir < m_Ndim; ++dir) {
    vout.general(m_vl, "  boundary[%d] = %2d\n", dir, bc[dir]);
  }

  //- range check
  // NB. kappa = 0 is allowed.
  assert(bc.size() == m_Ndim);

  //- store values
  m_kappa = kappa;

  m_boundary.resize(m_Ndim);
  for (int mu = 0; mu < m_Ndim; ++mu) {
    m_boundary[mu] = bc[mu];
  }

  //- propagate parameters
  m_fopr_w->set_parameters(m_kappa, m_boundary);
}


//====================================================================
void Fopr_Wilson_SF::DdagD(Field& w, const Field& f)
{
  Field w2(f.nin(), f.nvol(), f.nex());

  D(w2, f);
  mult_gm5(w, w2);
  D(w2, w);
  mult_gm5(w, w2);
}


//====================================================================
void Fopr_Wilson_SF::Ddag(Field& w, const Field& f)
{
  Field w2(f.nin(), f.nvol(), f.nex());

  mult_gm5(w, f);
  D(w2, w);
  mult_gm5(w, w2);
}


//====================================================================
void Fopr_Wilson_SF::H(Field& w, const Field& f)
{
  Field w2(f.nin(), f.nvol(), f.nex());

  D(w2, f);
  mult_gm5(w, w2);
}


//====================================================================
void Fopr_Wilson_SF::D(Field& w, const Field& f)
{
  Field w2(f);

  set_zero.set_boundary_zero(w2);
  m_fopr_w->D(w, w2);
  set_zero.set_boundary_zero(w);
}


//====================================================================
const Field_F Fopr_Wilson_SF::DdagD(const Field_F& f)
{
  Field_F w = H(H(f));

  return w;
}


//====================================================================
const Field_F Fopr_Wilson_SF::Ddag(const Field_F& f)
{
  Field_F w(f);

  w = (Field_F)mult_gm5((Field)w);
  w = H(w);

  return w;
}


//====================================================================
const Field_F Fopr_Wilson_SF::H(const Field_F& f)
{
  Field_F w = D(f);

  w = (Field_F)mult_gm5((Field)w);

  return w;
}


//====================================================================
const Field_F Fopr_Wilson_SF::D(const Field_F& f)
{
  int     Nvol = f.nvol();
  int     Nex  = f.nex();
  Field_F w(Nvol, Nex);

  Field_F w2(f);

  set_zero.set_boundary_zero(w2);

  //  w = m_fopr_w->D(f);
  m_fopr_w->set_mode("D");
  w = (Field_F)m_fopr_w->mult(w2);
  set_zero.set_boundary_zero(w);

  return w;
}


//====================================================================
const Field_F Fopr_Wilson_SF::mult_gm5p(int mu, const Field_F& w)
{
  Field_F vt(w.nvol(), w.nex());
  Field_F w2(w);

  set_zero.set_boundary_zero(w2);
  vt = m_fopr_w->mult_gm5p(mu, w2);
  set_zero.set_boundary_zero(vt);

  return vt;
}


//====================================================================

/*!
  Set the boundary field to zero: \f$\psi(t=0,\vec{x})=0\f$
 */

/*
void Fopr_Wilson_SF::set_boundary_zero(Field& f){
  if(comm->ipe(3)==0){
    for(int site = 0; site < Svol; ++site){
      for(int s = 0; s < m_Nd; ++s){
        for(int cc = 0; cc < m_Nc2; ++cc){
          f.set(cc+m_Nc2*s, site, 0, 0.0);
        }
      }
    }
  }
}
*/

//====================================================================
//============================================================END=====
