/*!
        @file    $Id:: fopr_Wilson_SF.h #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/


#ifndef FOPR_WILSON_SF_INCLUDED
#define FOPR_WILSON_SF_INCLUDED

#include "fopr_Wilson.h"

#include "field_F_SF.h"

//#include "index_lex.h"
//#include "shiftField_lex.h"

#include "bridgeIO.h"
using Bridge::vout;

//- parameters class
class Parameters_Fopr_Wilson_SF : virtual public Parameters
{
 public:
  Parameters_Fopr_Wilson_SF();
};

//! Wilson fermion operator with SF BC.

/*!
    This class implements the Wilson fermion operator with the SF BC.
    <ul>
    <li>The SF BC for the fermion field is:
    <ul>
    <li>The fermion field at t=0 is always set to zero (Dirichlet BC).
    <li>The field at t=1,...,Lt-1 is active.
    </ul>
    <li>Implemented by delegation of the Fopr_Wilson class like Fopr_Clover.
    <li>The modification is only in Fopr_Wilson_SF::D to set the boundary fermion field to zero before and after multiplication of the standard Wilson Dirac operator.
    <ul>
    <li>By this manipulation the SF BC is introduced in the fermion field.
    </ul>
    <li>A private function set_boundary_zero(Field&) is introduced for this bounadry manipulation.
    <li>A few private members are added for this function.
    <li> [04 Apr 2012 Y.Taniguchi]
    </ul>
    (Coding history will be recovered from trac.)
    YAML is implemented.             [14 Nov 2012 Y.Namekawa]
 */

class Fopr_Wilson_SF : public Fopr
{
 private:
  int                m_Nvol, m_Ndim, m_Nc, m_Nd, m_NinF;
  double             m_kappa;
  std::valarray<int> m_boundary;
  std::string        m_mode;

  Fopr_Wilson   *m_fopr_w;
  const Field_G *m_U;

  // Index_lex          m_idx;
  // ShiftField_lex     m_shift;

  /*
  //! Needed to know a node at the temporal boundary.
  Communicator* comm;
  //! A spatial volume in a node.
  int Svol;
  //! num of the double color elements
  int m_Nc2;
  */

  //! In order to set the boundary field to zero.
  Field_F_SF set_zero;

 public:

  Fopr_Wilson_SF()
  {
    m_Nvol = CommonParameters::Nvol();
    m_Ndim = CommonParameters::Ndim();
    m_Nc   = CommonParameters::Nc();
    m_Nd   = CommonParameters::Nd();
    m_NinF = 2 * m_Nc * m_Nd;

    m_boundary.resize(m_Ndim);
    m_U      = 0;
    m_fopr_w = new Fopr_Wilson;

    /*
    comm = Communicator::init();
    m_Nc2 = 2*m_Nc;
    int Nt = CommonParameters::Nt();
    Svol=m_Nvol/Nt;
    */
  }

  void set_parameters(const Parameters& params);
  void set_parameters(const double kappa, const std::valarray<int> bc);

  void set_config(Field *U)
  {
    m_U = (Field_G *)U;
    m_fopr_w->set_config(U);
  }

  ~Fopr_Wilson_SF()
  {
    delete m_fopr_w;
  }

  const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  void mult(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      D(v, f);
    } else if (m_mode == "DdagD") {
      DdagD(v, f);
    } else if (m_mode == "Ddag") {
      Ddag(v, f);
    } else if (m_mode == "H") {
      H(v, f);
    } else {
      vout.crucial(m_vl, "Fopr_Wilson_SF: mode undefined.\n");
      abort();
    }
  }

  void mult_dag(Field& v, const Field& f)
  {
    if (m_mode == "D") {
      Ddag(v, f);
    } else if (m_mode == "DdagD") {
      DdagD(v, f);
    } else if (m_mode == "Ddag") {
      D(v, f);
    } else if (m_mode == "H") {
      H(v, f);
    } else {
      vout.crucial(m_vl, "Fopr_Wilson_SF: mode undefined.\n");
      abort();
    }
  }

  void set_mode(std::string mode)
  {
    m_mode = mode;
  }

  std::string get_mode() const
  {
    return m_mode;
  }

  const Field_F DdagD(const Field_F&);
  const Field_F D(const Field_F&);
  const Field_F Ddag(const Field_F&);
  const Field_F H(const Field_F&);

  const Field mult_gm5(const Field& w)
  {
    return m_fopr_w->mult_gm5(w);
  }

  void DdagD(Field&, const Field&);
  void D(Field&, const Field&);
  void Ddag(Field&, const Field&);
  void H(Field&, const Field&);

  void mult_gm5(Field& v, const Field& w)
  {
    m_fopr_w->mult_gm5(v, w);
  }

  const Field_F mult_gm5p(int mu, const Field_F& w);

  int field_nvol() { return CommonParameters::Nvol(); }
  int field_nin()
  {
    return 2 * CommonParameters::Nc() * CommonParameters::Nd();
  }

  int field_nex() { return 1; }

 private:

  // A function to set the fermion field to zero at the t=0 boundary.
  //  void set_boundary_zero(Field&);
};
#endif
