/*!
        @file    $Id:: fopr_Wilson_eo.h #$

        @brief

        @author  UEDA, Satoru <sueda@post.kek.jp>
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/


#ifndef FOPR_WILSON_EO_INCLUDED
#define FOPR_WILSON_EO_INCLUDED

#include "fopr_Wilson.h"

#include "fopr_eo.h"
#include "shiftField_eo.h"

#include "gammaMatrixSet.h"

#include "bridgeIO.h"
using Bridge::vout;

//- parameters class
class Parameters_Fopr_Wilson_eo : virtual public Parameters
{
 public:
  Parameters_Fopr_Wilson_eo();
};
//- end

//! Even-odd Wilson fermion operator.

/*!
    This class is an even-odd version of Wilson fermion operator.
    At present this is rough implementation, while correctly
    works, and to be updated by supplying complete functionality.
    Only the functions needed for even-odd preconditioned solver
    is ready.
                                     [20 Jun 2012 S.UEDA]
    (Coding history will be recovered from trac.)
    YAML is implemented.             [14 Nov 2012 Y.Namekawa]
    Selector is implemented.         [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

class Fopr_Wilson_eo : public Fopr_eo
{
 private:
  int                m_Nvol, m_Nvol2, m_Ndim;
  double             m_kappa;
  std::valarray<int> m_boundary;
  Index_eo           m_index;
  Field_G            *m_Ueo;

  ShiftField_eo shift;
  Field_F       trf, trf2;
  Vec_SU_N      v1, v2;

  std::valarray<GammaMatrix> m_GM;

  std::string m_mode;
  std::string m_repr;

  void (Fopr_Wilson_eo::*m_mult)     (Field&, const Field&);
  void (Fopr_Wilson_eo::*m_mult_dag) (Field&, const Field&);
  void (Fopr_Wilson_eo::*m_D)        (Field&, const Field&);
  void (Fopr_Wilson_eo::*m_gm5)      (Field&, const Field&);
  void (Fopr_Wilson_eo::*m_preProp)  (Field&, Field&, const Field&);
  void (Fopr_Wilson_eo::*m_postProp) (Field&, const Field&, const Field&);

 public:

  Fopr_Wilson_eo(std::string repr)
    : m_Nvol(CommonParameters::Nvol()),
      m_Nvol2(CommonParameters::Nvol() / 2),
      m_Ndim(CommonParameters::Ndim()),
      trf(m_Nvol2, 1), trf2(m_Nvol2, 1)
  {
    init(repr);
  }

  ~Fopr_Wilson_eo()
  {
    delete m_Ueo;
  }

  void set_parameters(const Parameters& params);
  void set_parameters(const double kappa, const std::valarray<int> bc);

  void set_config(Field *U);

  // method for even odd fermion operator
  void preProp(Field& Be, Field& bo, const Field& b)
  {
    (this->*m_preProp)(Be, bo, b);
  }

  void postProp(Field& x, const Field& xe, const Field& bo)
  {
    (this->*m_postProp)(x, xe, bo);
  }

  void prePropD(Field&, Field&, const Field&);
  void postPropD(Field&, const Field&, const Field&);
  void prePropDag(Field&, Field&, const Field&);
  void postPropDag(Field&, const Field&, const Field&);

  const Field mult(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult(v, f);
    return v;
  }

  const Field mult_dag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    mult_dag(v, f);
    return v;
  }

  void mult(Field& v, const Field& f)
  {
    (this->*m_mult)(v, f);
  }

  void mult_dag(Field& v, const Field& f)
  {
    (this->*m_mult_dag)(v, f);
  }

  void set_mode(std::string mode)
  {
    m_mode = mode;

    if (m_mode == "D") {
      m_mult     = &Fopr_Wilson_eo::D;
      m_mult_dag = &Fopr_Wilson_eo::Ddag;
      m_preProp  = &Fopr_Wilson_eo::prePropD;
      m_postProp = &Fopr_Wilson_eo::postPropD;
    } else if (m_mode == "Ddag") {
      m_mult     = &Fopr_Wilson_eo::Ddag;
      m_mult_dag = &Fopr_Wilson_eo::D;
      m_preProp  = &Fopr_Wilson_eo::prePropDag;
      m_postProp = &Fopr_Wilson_eo::postPropDag;
    } else if (m_mode == "DdagD") {
      m_mult     = &Fopr_Wilson_eo::DdagD;
      m_mult_dag = &Fopr_Wilson_eo::DdagD;
    } else if (m_mode == "H") {
      m_mult     = &Fopr_Wilson_eo::H;
      m_mult_dag = &Fopr_Wilson_eo::H;
    } else {
      vout.crucial("Fopr_Wilson_eo: input mode is undefined.\n");
      abort();
    }
  }

  std::string get_mode() const
  {
    return m_mode;
  }

  void mult_undef(Field&, const Field&)
  {
    vout.crucial("Fopr_Wilson_eo: mode undefined.\n");
    abort();
  }

  const Field MeoMoe(const Field& f)
  {
    Field_F v((Field_F)f);

    v -= Meo(Meo(f, 1), 0);
    return (Field)v;
  }

  const Field D(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    D(v, f);
    return v;
  }

  void D(Field& v, const Field& f)
  {
    v  = f;
    v -= (Field)Meo(Meo(f, 1), 0);
  }

  const Field Ddag(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    Ddag(v, f);
    return v;
  }

  void Ddag(Field& v, const Field& f)
  {
    v  = f;
    v -= (Field)Mdageo(Mdageo(f, 1), 0);
  }

  const Field DdagD(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    DdagD(v, f);
    return v;
  }

  void DdagD(Field& v, const Field& f)
  {
    v = Ddag(D(f));
  }

  const Field H(const Field& f)
  {
    Field v(f.nin(), f.nvol(), f.nex());

    H(v, f);
    return v;
  }

  void H(Field& v, const Field& f)
  {
    v = mult_gm5(D(f));
  }

  //  const Field_F     D(const Field_F&);
  //  const Field_F     H(const Field_F&);

  const Field_F Meo(const Field_F&, const int ieo);
  const Field_F Mdageo(const Field_F&, const int ieo);

  // ieo=0: even <-- odd
  // ieo=1: odd  <-- even

  const Field_F Meo_gm5(const Field_F&, const int ieo);

  const Field mult_gm5(const Field&);

  //! gamma_5 (1 - gamma_mu) v(x + mu)
  const Field gm5p(const int mu, const Field& v);

  int field_nvol() { return CommonParameters::Nvol() / 2; }
  int field_nin() { return 2 * CommonParameters::Nc() * CommonParameters::Nd(); }
  int field_nex() { return 1; }

 private:
  void init(const std::string);
  void mult_p(int mu, Field_F&, const Field_F&, const int ieo);
  void mult_m(int mu, Field_F&, const Field_F&, const int ieo);

  void mult_xp(Field_F&, const Field_F&, const int ieo);
  void mult_xm(Field_F&, const Field_F&, const int ieo);
  void mult_yp(Field_F&, const Field_F&, const int ieo);
  void mult_ym(Field_F&, const Field_F&, const int ieo);
  void mult_zp(Field_F&, const Field_F&, const int ieo);
  void mult_zm(Field_F&, const Field_F&, const int ieo);
  void mult_tp(Field_F&, const Field_F&, const int ieo);
  void mult_tm(Field_F&, const Field_F&, const int ieo);
};
#endif
