/*!
        @file    $Id:: force_F_Clover_Nf2.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "force_F_Clover_Nf2.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("hopping_parameter", 0.0);
    param.Register_double("clover_coefficient", 0.0);
    param.Register_int_vector("boundary_condition", std::valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Force.F_Clover_Nf2", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Force_F_Clover_Nf2::Parameters_Force_F_Clover_Nf2() { append_entry(*this); }
//- end

//====================================================================
void Force_F_Clover_Nf2::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double        kappa, cSW;
  valarray<int> bc;

  int err = 0;
  err += params.fetch_double("hopping_parameter", kappa);
  err += params.fetch_double("clover_coefficient", cSW);
  err += params.fetch_int_vector("boundary_condition", bc);

  if (err) {
    vout.crucial(m_vl, "Force_F_Clover_Nf2: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(kappa, cSW, bc);
}


//====================================================================
void Force_F_Clover_Nf2::set_parameters(const double kappa, const double cSW,
                                        const valarray<int> bc)
{
  int Ndim = CommonParameters::Ndim();

  //- print input parameters
  vout.general(m_vl, "Parameters of Force_F_Clover_Nf2:\n");
  vout.general(m_vl, "  kappa = %8.4f\n", kappa);
  vout.general(m_vl, "  cSW   = %8.4f\n", cSW);
  for (int mu = 0; mu < Ndim; ++mu) {
    vout.general(m_vl, "  boundary[%d] = %2d\n", mu, bc[mu]);
  }

  //- range check
  // NB. kappa,cSW == 0 is allowed.
  assert(bc.size() == Ndim);

  //- store values
  m_kappa = kappa;
  m_cSW   = cSW;

  m_boundary.resize(Ndim);
  for (int mu = 0; mu < Ndim; ++mu) {
    m_boundary[mu] = bc[mu];
  }

  //- propagate parameters
  m_fopr_c->set_parameters(m_kappa, m_cSW, m_boundary);

  m_force_w->set_parameters(m_kappa, m_boundary);
  m_force_csw->set_parameters(m_kappa, m_cSW, m_boundary);
}


//====================================================================
void Force_F_Clover_Nf2::init(std::string repr)
{
  m_repr = repr;

  m_fopr_c    = new Fopr_Clover(repr);
  m_force_w   = new Force_F_Wilson_Nf2(repr);
  m_force_csw = new Force_F_CloverTerm(repr);

  m_boundary.resize(CommonParameters::Ndim());

  int Nvol = CommonParameters::Nvol();
  m_Ndim = CommonParameters::Ndim();

  m_Cud = new Field_G(Nvol, m_Ndim * m_Ndim);
}


//====================================================================
void Force_F_Clover_Nf2::tidyup()
{
  delete m_Cud;
  delete m_force_csw;
  delete m_force_w;
  delete m_fopr_c;
}


//====================================================================
Field Force_F_Clover_Nf2::force_core(const Field& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  Field_G  force(Nvol, Ndim), force1(Nvol, Ndim);
  Mat_SU_N ut(Nc);

  force1 = force_udiv(eta);

  for (int mu = 0; mu < Ndim; ++mu) {
    force.mult_Field_Gnn(mu, *m_U, mu, force1, mu);
    force.at_Field_G(mu);
  }
  force *= -2.0;

  return (Field)force;
}


//====================================================================
Field Force_F_Clover_Nf2::force_core1(const Field& zeta,
                                      const Field& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  Field_G  force(Nvol, Ndim), force1(Nvol, Ndim);
  Mat_SU_N ut(Nc);

  force1 = force_udiv1(zeta, eta);

  for (int mu = 0; mu < Ndim; ++mu) {
    force.mult_Field_Gnn(mu, *m_U, mu, force1, mu);
    force.at_Field_G(mu);
  }
  force *= -2.0;

  return (Field)force;
}


//====================================================================
Field Force_F_Clover_Nf2::force_udiv(const Field& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  Field force(NinG, Nvol, Ndim);

  Field_F zeta(Nvol, 1);

  //  zeta = m_fopr_c->H(eta);
  m_fopr_c->H(zeta, eta);

  force  = force_udiv1((Field_F)eta, zeta);
  force += force_udiv1(zeta, (Field_F)eta);

  return force;
}


//====================================================================
Field Force_F_Clover_Nf2::force_udiv1(const Field& zeta,
                                      const Field& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  Field force(NinG, Nvol, Ndim);

  force = force_udiv1_impl((Field_F)zeta, (Field_F)eta);

  return force;
}


//====================================================================
Field Force_F_Clover_Nf2::force_udiv1_impl(const Field_F& zeta,
                                           const Field_F& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nd   = CommonParameters::Nd();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  ShiftField_lex shift;

  Field force(NinG, Nvol, Ndim);
  Field force2(NinG, Nvol, Ndim);
  //  Field_G force1(Nvol,1), force2(Nvol,1);
  Field_G Umu(Nvol, 1), Unu(Nvol, 1), Utmp(Nvol, 1), Utmp2(Nvol, 1);
  Field_F vt1(Nvol, 1), vt2(Nvol, 1), vt3(Nvol, 1), vt4(Nvol, 1);
  Field_F zeta_mu(Nvol, 1);

  Mat_SU_N ut(Nc);
  Vec_SU_N vec1(Nc), vec2(Nc);

  force  = m_force_w->force_udiv1(zeta, eta);
  force2 = m_force_csw->force_udiv1(zeta, eta);

  force += force2;

  /*
  for(int dir = 0; dir < Ndim; ++dir){
    force.addpart_ex(dir,force2,dir);
  }
  */

  /*
  Field_F eta2(Nvol,1), eta3(Nvol,1);
  eta2 = (Field_F)m_fopr_c->mult_gm5(eta);

  for(int mu = 0; mu < Ndim; ++mu){
   for(int nu = 0; nu < Ndim; ++nu){
     if(nu==mu) continue;

     m_fopr_c->mult_isigma(eta3,eta2,mu,nu);

     Umu.setpart_ex(0, *m_U,mu);
     Unu.setpart_ex(0, *m_U,nu);

     int ex = 0;
     // R(1) and R(5)
     vt1.mult_Field_Gd(0,*m_Cud,index_dir(mu,nu),eta3,ex);
     tensorProd_Field_F(force1,zeta,vt1);
     force2 = force1;

     // R(2)
     vt3.mult_Field_Gd(0,Umu,0,eta3,ex);
     shift.backward(vt1,vt3,nu);
     shift.backward(vt2,zeta,nu);
     shift.backward(Utmp,Unu,mu);
     vt3.mult_Field_Gn(0,Utmp,0,vt1,ex);
     vt4.mult_Field_Gn(0,Unu,0,vt2,ex);
     tensorProd_Field_F(force1,vt4,vt3);
     force2 += force1;

     // R(4) and R(8)
     shift.backward(vt1,eta3,mu);
     shift.backward(zeta_mu,zeta,mu);
     vt4.mult_Field_Gn(0,*m_Cud,index_dir(mu,nu),zeta_mu,ex);
     tensorProd_Field_F(force1,vt4,vt1);
     force2 += force1;

     // R(3)
     shift.backward(vt1,eta3,nu);
     vt3.mult_Field_Gn(0,Unu,0,vt1,ex);
     vt4.mult_Field_Gn(0,Umu,0,zeta_mu,ex);
     shift.backward(vt1,vt3,mu);
     shift.backward(vt2,vt4,nu);
     vt4.mult_Field_Gn(0,Unu,0,vt2,ex);
     tensorProd_Field_F(force1,vt4,vt1);
     force2 += force1;

     // R(6)
     shift.backward(Utmp,Unu,mu);
     Utmp2.mult_Field_Gdd(0,Utmp,0,Umu,0);
     vt1.mult_Field_Gn(0,Utmp2,0,eta3,ex);
     vt2.mult_Field_Gd(0,Unu,0,zeta,ex);
     shift.forward(vt3,vt1,nu);
     shift.forward(vt4,vt2,nu);
     tensorProd_Field_F(force1,vt4,vt3);
     force2 -= force1;

     // R(7)
     vt1.mult_Field_Gd(0,Unu,0,eta3,ex);
     vt2.mult_Field_Gn(0,Umu,0,zeta_mu,ex);
     shift.backward(vt3,vt1,mu);
     shift.forward(vt1,vt3,nu);
     vt4.mult_Field_Gd(0,Unu,0,vt2,ex);
     shift.forward(vt2,vt4,nu);
     tensorProd_Field_F(force1,vt2,vt1);
     force2 -= force1;

     force2 *= -m_kappa * m_cSW /8.0;
     force.addpart_ex(mu, force2,0);

   }
  }
  */

  return force;
}


//====================================================================
void Force_F_Clover_Nf2::set_component()
{
  int Nc   = CommonParameters::Nc();
  int Nd   = CommonParameters::Nd();
  int Nvol = CommonParameters::Nvol();

  Staples staple;
  Field_G Cmu_ud(Nvol, 1);

  for (int mu = 0; mu < m_Ndim; ++mu) {
    for (int nu = 0; nu < m_Ndim; ++nu) {
      if (nu == mu) continue;

      Cmu_ud  = staple.upper(*m_U, mu, nu);
      Cmu_ud -= staple.lower(*m_U, mu, nu);
      m_Cud->setpart_ex(index_dir(mu, nu), Cmu_ud, 0);
    }
  }
}


//====================================================================
//============================================================END=====
