/*!
        @file    $Id:: force_F_Wilson_eo.cpp#$

        @brief

        @author  <UEDA, Satoru> sueda@post.kek.jp (sueda)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "force_F_Wilson_eo.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using Bridge::vout;

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("hopping_parameter", 0.0);
    param.Register_int_vector("boundary_condition", std::valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Force.F_Wilson_eo", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Force_F_Wilson_eo::Parameters_Force_F_Wilson_eo() { append_entry(*this); }
//- end

//====================================================================
void Force_F_Wilson_eo::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double        kappa;
  valarray<int> bc;

  int err = 0;
  err += params.fetch_double("hopping_parameter", kappa);
  err += params.fetch_int_vector("boundary_condition", bc);

  if (err) {
    vout.crucial(m_vl, "Force_F_Wilson_eo: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(kappa, bc);
}


//====================================================================
void Force_F_Wilson_eo::set_parameters(double kappa, const std::valarray<int> bc)
{
  int Ndim = CommonParameters::Ndim();

  //- print input parameters
  vout.general(m_vl, "Parameters of Force_F_Wilson_eo:\n");
  vout.general(m_vl, "  kappa = %8.4f\n", kappa);
  for (int mu = 0; mu < Ndim; ++mu) {
    vout.general(m_vl, "  boundary[%d] = %2d\n", mu, bc[mu]);
  }

  //- range check
  // NB. kappa == 0 is allowed.
  assert(bc.size() == Ndim);

  //- store values
  m_kappa = kappa;

  m_boundary.resize(Ndim);
  for (int mu = 0; mu < Ndim; ++mu) {
    m_boundary[mu] = bc[mu];
  }

  //- post-process
  m_fopr_w->set_parameters(m_kappa, m_boundary);
}


//====================================================================
Field Force_F_Wilson_eo::force_core(const Field& eta_e)
{
  int Nc    = CommonParameters::Nc();
  int Nvol  = CommonParameters::Nvol();
  int Nvol2 = Nvol / 2;
  int Ndim  = CommonParameters::Ndim();
  int NinG  = 2 * Nc * Nc;

  Field_G force(Nvol, Ndim);
  Field_G force1(Nvol, Ndim);

  force1 = force_udiv(eta_e);

  for (int mu = 0; mu < Ndim; ++mu) {
    force.mult_Field_Gnn(mu, *m_U, mu, force1, mu);
    force.at_Field_G(mu);
  }
  force *= -2.0;

  return (Field)force;
}


//====================================================================
Field Force_F_Wilson_eo::force_core1(const Field& zeta,
                                     const Field& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();
  int NinG = 2 * Nc * Nc;

  Field_G force(Nvol, Ndim);
  Field_G force1(Nvol, Ndim);

  force1 = force_udiv1(zeta, eta);

  for (int mu = 0; mu < Ndim; ++mu) {
    force.mult_Field_Gnn(mu, *m_U, mu, force1, mu);
    force.at_Field_G(mu);
  }
  force *= -2.0;

  return (Field)force;
}


//====================================================================
Field Force_F_Wilson_eo::force_udiv(const Field& eta_e)
{
  int Nc    = CommonParameters::Nc();
  int Nd    = CommonParameters::Nd();
  int Nvol  = CommonParameters::Nvol();
  int Ndim  = CommonParameters::Ndim();
  int Nvol2 = Nvol / 2;

  Field_G force(Nvol, Ndim);
  Field_G force1(Nvol, Ndim);

  Field_F eta_o(Nvol2, 1);
  Field_F eta(Nvol, 1);

  eta_o = m_fopr_w->Meo(eta_e, 1);
  m_index.mergeField(eta, eta_e, eta_o);

  Field_F zeta_e(Nvol2, 1);
  Field_F zeta_o(Nvol2, 1);
  Field_F zeta(Nvol, 1);

  //zeta = m_fopr_w->H(eta);
  m_fopr_w->set_mode("H");
  zeta_e = (Field_F)m_fopr_w->mult(eta_e);
  zeta_o = m_fopr_w->Meo(zeta_e, 1);
  m_index.mergeField(zeta, zeta_e, zeta_o);

  force  = (Field_G)force_udiv1(zeta, (Field_F)eta);
  force1 = (Field_G)force_udiv1((Field_F)eta, zeta);
  force += force1;

  return (Field)force;
}


//====================================================================
Field Force_F_Wilson_eo::force_udiv1(const Field& zeta,
                                     const Field& eta)
{
  return force_udiv1((Field_F)zeta, (Field_F)eta);
}


//====================================================================
Field Force_F_Wilson_eo::force_udiv1(const Field_F& zeta,
                                     const Field_F& eta)
{
  int Nc   = CommonParameters::Nc();
  int Nd   = CommonParameters::Nd();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();

  Field_G force(Nvol, Ndim);
  Field_G force1(Nvol, 1);

  Field_F eta2(Nvol, 1);

  for (int mu = 0; mu < Ndim; ++mu) {
    eta2 = (Field_F)m_fopr_w->gm5p(mu, eta);
    tensorProd_Field_F(force1, zeta, eta2);
    force.setpart_ex(mu, force1, 0);
  }

  force *= -m_kappa;

  return (Field)force;
}


//====================================================================
//============================================================END=====
