/*!
        @file    $Id:: builder_Integrator.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#ifndef BUILDER_INTEGRATOR_INCLUDED
#define BUILDER_INTEGRATOR_INCLUDED

#include <cassert>

#include "integrator_UpdateU.h"

#include "integrator_Leapfrog.h"
#include "integrator_Omelyan.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Builder of MD integrator for HMC.

/*!
    This class builds a molecular dynamics integrator for hybrid Monte
    Carlo update algorithm by composing subclasses of Integrator.
    At present standard leapfrog and omelyan integrators can be
    composed of.
                                     [25 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.
                                     [03 Mar 2013 Y.Namekawa]
 */

//- parameters class
class Parameters_Builder_Integrator : virtual public Parameters
{
 public:
  Parameters_Builder_Integrator();
};
//- end

class Builder_Integrator
{
 protected:
  Bridge::VerboseLevel m_vl;

 private:
  double                      m_Estep; //!< step size of each molecular dynamical evolution
  int                         m_Nprec; //!< precision parameter of exponentiation
  double                      m_lambda_Omelyan;
  std::string                 m_str_integrator_type;
  std::valarray<Action *>     m_action;
  std::valarray<Director *>   m_director;
  int                         m_Nlevel;  //!< Number of levels
  std::valarray<int>          m_Naction; //!< Number of actions at each level
  std::valarray<int>          m_Nstep;   //!< Number of steps at each level
  std::valarray<Integrator *> m_integs;  //!< Integrator to be constructed

 public:
  //! constructor with action and director.
  Builder_Integrator(std::valarray<Action *> action,
                     std::valarray<Director *> director)
    : m_vl(CommonParameters::Vlevel())
  {
    m_action.resize(action.size());
    for (int i = 0; i < action.size(); ++i) {
      m_action[i] = action[i];
    }
    m_director.resize(director.size());
    for (int i = 0; i < director.size(); ++i) {
      m_director[i] = director[i];
    }
    m_Estep          = 0.0;
    m_Nprec          = 0;
    m_lambda_Omelyan = 0.0;
  }

  //! constructor when no director is necessary.
  Builder_Integrator(std::valarray<Action *> action)
    : m_vl(CommonParameters::Vlevel())
  {
    m_action.resize(action.size());
    for (int i = 0; i < action.size(); ++i) {
      m_action[i] = action[i];
    }
    m_Estep          = 0.0;
    m_Nprec          = 0;
    m_lambda_Omelyan = 0.0;
  }

  //! destructor
  ~Builder_Integrator()
  {
    tidyup();
  }

  void set_parameters(const Parameters& params);
  void set_parameters(std::string str_integrator_type,
                      double Estep, int Nlevel,
                      const std::valarray<int>& Naction,
                      const std::valarray<int>& Nstep,
                      const int Nprec,
                      const double lambda_Omelyan);

  void set_parameter_verboselevel(const Bridge::VerboseLevel vl) { m_vl = vl; }

  Integrator *build();

  Integrator *build_leapfrog();
  Integrator *build_omelyan();

  void tidyup();
};
#endif
