/*!
        @file    $Id: fieldIO.h 943 2013-07-22 07:55:30Z sueda $
        @brief
        @author  $LastChangedBy: sueda $
        @date    $LastChangedDate: 2013-07-22 16:55:30 +0900 (月, 22  7月 2013) $
        @version $LastChangedRevision: 943 $
*/


#ifndef FIELDIO_INCLUDED
#define FIELDIO_INCLUDED

#include <string>
#include <stdint.h>

#include "field.h"
#include "index_lex.h"
#include "commonParameters.h"
#include "io_format.h"

//! FieldIO class for file I/O of space-time distributed data.

/*!
   This class family is used to setup and output the gauge
   configuration.
   This is the base class, which implements common functions,
   and read/write methods are implemented in subclasses.
   At present, cutting off the gauge field for each node
   and deliver it to the node is implemented in this class.
   It may be better to separate that to other class for
   general usage for other field objects.
                                [28 Dec 2011 H.Matsufuru]

   FieldIO class provides file I/O of Field data.
   Reading and writing gauge configuration from/to file is now
   implemented on top of this construct.

   For scalar data (independent of space-time index), DataIO class
   hierarchy is provided.

   FieldIO class is an abstract base class; various data format
   (text, binary, etc) and I/O scheme (single node or parallel I/O)
   are realised in the respective subclasses.
   This class also provides common utilities for gathering/distributing
   data over parallel nodes (if any), and converting byte order.
*/

class FieldIO
{
 private:
  Index_lex idx;

 protected:
  const IO_Format::Format *m_format;

  Bridge::VerboseLevel m_vl;

 public:

  //!< constructor. format specifies data layout on file.
  FieldIO(const IO_Format::Format *format) : m_format(format), m_vl(CommonParameters::Vlevel()) {}
  virtual ~FieldIO() {}

  virtual void read_file(Field *v, std::string)  = 0; //!< read data from file.
  virtual void write_file(Field *v, std::string) = 0; //!< write data to file.

  void deliver(Field *vlocal, Field *vglobal);        //!< distribute data on primary node over parallel nodes.
  void gather(Field *vglobal, Field *vlocal);         //!< gather data on parallel nodes to primary node.

 protected:

  //!< convert byte order. alternative interface.
  void byte_swap(void *buf, size_t size, size_t nmemb)
  {
    return convert_endian(buf, size, nmemb);
  }

  //!< convert byte order of data, each of whose element has size bytes.
  void convert_endian(void *buf, size_t size, size_t nmemb);

  //!< check if machine byte order is big-endian.
  bool is_bigendian() const;
};
#endif
