/*!
        @file    $Id: fieldIO_Fortran.cpp 930 2013-07-12 07:56:41Z sueda $
        @brief
        @author  $LastChangedBy: sueda $
        @date    $LastChangedDate: 2013-07-12 16:56:41 +0900 (金, 12  7月 2013) $
        @version $LastChangedRevision: 930 $
*/


// This class is to read binary file generated by Fortran code.
// If the endian of the system differs from file format,
//   byte_swap must be called.

#include "fieldIO_Fortran.h"

#include <iostream>
#include <fstream>

#include "commonParameters.h"
#include "communicator.h"
#include "bridgeIO.h"
using Bridge::vout;

#include "field.h"

//====================================================================
void FieldIO_Fortran::read_file(Field *v, std::string filename)
{
  int nin_field = v->nin();
  int nex_field = v->nex();

  int nin_file = m_format->nin();
  int nex_file = m_format->nex();

  if ((nin_file == 0) || (nex_file == 0)) {
    nin_file = nin_field;
    nex_file = nex_field;
  }

  int Lvol = CommonParameters::Lvol();

  int size = nin_file * Lvol * nex_file;

  Field vtmp;

  if (Communicator::is_primary()) {
    vout.detailed(m_vl, "reading field data from %s", filename.c_str());

    vtmp.reset(nin_field, Lvol, nex_field);

    std::fstream infile;
    infile.open(filename.c_str(), std::ios::in | std::ios::binary);

    bool do_byteswap = false;

    // Skipping 4B data size area (for size less than 2 GB)
    // infile.seekg(4);

    // read header: record length (bytes)
    // assume header is 4bytes.
    uint32_t flen;
    infile.read((char *)&flen, sizeof(uint32_t) * 1);
    if (!infile) {
      vout.general(m_vl, "file = %s open failed\n", filename.c_str());
      return;
    }

    // find byteorder
    if (flen == sizeof(double) * size) {
      // file byteorder is equal to machine byteorder
      vout.detailed(m_vl, "endian matched.\n");
    } else {
      uint32_t flen_s = flen;
      byte_swap(&flen_s, sizeof(uint32_t), 1);

      if (flen_s == sizeof(double) * size) {
        // file byteorder is different from machine byteorder: need swap.
        vout.detailed(m_vl, "different endian. do swap.\n");
        do_byteswap = true;
      } else {
        vout.crucial(m_vl, "size mismatch or format unidentified.\n");
        abort();
      }
    }

    // read content
    const int block_size = nin_file;
    char      buf[sizeof(double) * block_size];

    for (int j = 0; j < nex_file; ++j) {
      for (int isite = 0; isite < Lvol; ++isite) {
        // read 1 block
        infile.read(buf, sizeof(double) * block_size);

        if (do_byteswap) {
          byte_swap(buf, sizeof(double), block_size);
        }

        double *ptr = (double *)buf;

        for (int i = 0; i < nin_file; ++i) {
          int s, t;
          m_format->file_to_field(s, t, i, j);

          vtmp.set(s, isite, t, ptr[i]);
        }
      }
    }

    // read trailer
    uint32_t ftail;
    infile.read((char *)&ftail, sizeof(uint32_t) * 1);

    if (flen != ftail) {
      vout.crucial(m_vl, "record info mismatch.\n");
      abort();
    }
    if (!infile) {
      vout.crucial(m_vl, "file read failed.\n");
      abort();
    }

    infile.close();
  }

  FieldIO::deliver(v, &vtmp);

  vout.detailed(m_vl, "read successful\n");
}


//====================================================================
void FieldIO_Fortran::write_file(Field *v, std::string filename)
{
  int nin_field = v->nin();
  int nex_field = v->nex();

  int nin_file = m_format->nin();
  int nex_file = m_format->nex();

  if ((nin_file == 0) || (nex_file == 0)) {
    nin_file = nin_field;
    nex_file = nex_field;
  }

  int Lvol = CommonParameters::Lvol();

  Field vtmp;
  if (Communicator::is_primary()) {
    vtmp.reset(nin_field, Lvol, nex_field);
  }

  size_t count = nin_file * Lvol * nex_file;

  FieldIO::gather(&vtmp, v);

  if (Communicator::is_primary()) {
    vout.detailed(m_vl, "writing field data to %s\n", filename.c_str());

    std::ofstream outfile(filename.c_str(), std::ios::out | std::ios::binary);
    if (!outfile) {
      vout.general(m_vl, "file = %s open failed\n", filename.c_str());
      return;
    }

    uint32_t flen = sizeof(double) * count;

    // record header: record length (bytes)
    outfile.write((char *)&flen, sizeof(uint32_t) * 1);

    // record content
    const int block_size = nin_file;
    char      buf[sizeof(double) * block_size];


    for (int j = 0; j < nex_file; ++j) {
      for (int isite = 0; isite < Lvol; ++isite) {
        double *ptr = (double *)buf;

        for (int i = 0; i < nin_file; ++i) {
          int s, t;
          m_format->file_to_field(s, t, i, j);

          ptr[i] = vtmp.cmp(s, isite, t);
        }

        outfile.write(buf, sizeof(double) * block_size);
      }
    }

    // record footer: record length (bytes)
    outfile.write((char *)&flen, sizeof(uint32_t) * 1);

    outfile.close();
  }

  vout.detailed(m_vl, "write succeeded.\n");
}


//====================================================================
//============================================================END=====
