/*!
        @file    $Id:: gaugeConfig.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "gaugeConfig.h"

#include "commonParameters.h"
#include "communicator.h"
#include "bridgeIO.h"
using Bridge::vout;

#include "field.h"
#include "mat_SU_N.h"

#include "fieldIO.h"
#include "fieldIO_Text.h"
#include "fieldIO_Binary.h"
#include "fieldIO_Binary_Parallel.h"
#include "fieldIO_Fortran.h"
#include "fieldIO_LIME.h"
#include "io_format.h"
#include "io_format_gauge.h"


//====================================================================
void GaugeConfig::set_cold(Field *U)
{
  int Nc   = CommonParameters::Nc();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();

  Field_G  *up = (Field_G *)U;
  Mat_SU_N ut(Nc);

  ut.unit();

  for (int mu = 0; mu < Ndim; ++mu) {
    for (int site = 0; site < Nvol; ++site) {
      up->set_mat(site, mu, ut);
    }
  }

  vout.detailed("gauge config. was set to unity.\n");
}


//====================================================================
GaugeConfig::GaugeConfig(const string& type)
  : m_prefix(), m_postfix(), m_vl(CommonParameters::Vlevel()), m_fieldio()
{
  if (type == "Text") {
    m_fieldio = new FieldIO_Text(IO_Format::Gauge::ILDG);
  } else if (type == "Binary") {
    m_fieldio = new FieldIO_Binary(IO_Format::Gauge::ILDG);
  } else if (type == "Fortran_JLQCD") {
    m_fieldio = new FieldIO_Fortran(IO_Format::Gauge::JLQCD);
  } else if (type == "Fortran_ILDG") {
    m_fieldio = new FieldIO_Fortran(IO_Format::Gauge::ILDG);
  } else if (type == "ILDG") {
    m_fieldio = new FieldIO_LIME(IO_Format::Gauge::ILDG);
  } else if (type == "Binary_Parallel") {
    m_fieldio = new FieldIO_Binary_Parallel(IO_Format::Gauge::ILDG);
  } else {
    vout.crucial("GaugeConfig: unsupported type \"%s\".\n", type.c_str());
    abort();
  }
}


//====================================================================
GaugeConfig::~GaugeConfig()
{
  if (m_fieldio) delete m_fieldio;
  m_fieldio = 0;
}


//====================================================================
string GaugeConfig::config_filename(const int confnum)
{
  const size_t filename_len = 256;
  char         buf[filename_len];

  snprintf(buf, filename_len, "%s%06d%s", m_prefix.c_str(), confnum, m_postfix.c_str());

  return string(buf);
}


//====================================================================
void GaugeConfig::read_file(Field *u, const string& filename)
{
  if (!m_fieldio) {
    vout.crucial("GaugeConfig::read_config(): FieldIO not set.");
    abort();
  }

  m_fieldio->read_file(u, filename);
}


//====================================================================
void GaugeConfig::write_file(Field *u, const string& filename)
{
  if (!m_fieldio) {
    vout.crucial("GaugeConfig::read_config(): FieldIO not set.");
    abort();
  }

  m_fieldio->write_file(u, filename);
}


//====================================================================
void GaugeConfig::read_config(Field *U, int confnum)
{
  return read_file(U, config_filename(confnum));
}


//====================================================================
void GaugeConfig::write_config(Field *U, int confnum)
{
  return write_file(U, config_filename(confnum));
}


//====================================================================
//============================================================END=====
