/*!
        @file    $Id:: corr2pt_4spinor.h#$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#ifndef CORR2PT_4SPINOR_INCLUDED
#define CORR2PT_4SPINOR_INCLUDED

#include <valarray>
#include "defs.h"
#include "commonParameters.h"
#include "index_lex.h"
#include "field_F.h"
#include "gammaMatrix.h"
#include "gammaMatrixSet.h"
#include "contract_4spinor.h"
#include "bridge_complex.h"
#include "bridgeIO.h"

//! Two-point correlator for Wilson-type fermions.

/*!
   So far meson correlators were implemented.
                                  [4 Feb 2012 H.Matsufuru]
   Baryon (proton) correlator was implemented by K.Nemuta.
   This implementation assumes Nc=3, and some of parameters
   are replaced by explicit numbers.
   Better performance version: [28 Jul 2012 H.Matsufuru].
 */

class Corr2pt_4spinor
{
 protected:
  Bridge::VerboseLevel m_vl;

 private:
  Index_lex          m_index;
  GammaMatrixSet     *m_gmset;
  std::valarray<int> m_epsilon_index;
  //!< index of totally antisymmetric tensor

 public:

  Corr2pt_4spinor(GammaMatrixSet *gmset)
    : m_vl(CommonParameters::Vlevel()), m_gmset(gmset)
  {
    setup();
  }

  void set_parameter_verboselevel(const Bridge::VerboseLevel vl) { m_vl = vl; }

  double meson_all(
    const std::valarray<Field_F>& sq1,
    const std::valarray<Field_F>& sq2);

  void meson_corr(
    std::valarray<dcomplex>& meson,
    const GammaMatrix& gm_sink,
    const GammaMatrix& gm_src,
    const std::valarray<Field_F>& sq1,
    const std::valarray<Field_F>& sq2);

  double proton_test(
    const std::valarray<Field_F>& squ,
    const std::valarray<Field_F>& sqd);

  void proton_corr(
    std::valarray<dcomplex>& proton,
    const GammaMatrix& gm,
    const std::valarray<Field_F>& squ,
    const std::valarray<Field_F>& sqd);

 private:

  void setup();

  //! totally antisymmetric tensor: index.
  int epsilon_index(int i, int n)
  {
    return m_epsilon_index[i + 3 * n];
  }

  //! totally antisymmetric tensor: value.
  double epsilon_value(int n)
  {
    return 1.0 - 2.0 * (n / 3);
  }

  //! transform node-local correlator in t to global.
  void global_corr_t(std::valarray<dcomplex>& corr_global,
                     std::valarray<dcomplex>& corr_local);
};
#endif
