/*!
        @file    $Id:: source_4spinor_Local.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: namekawa $

        @date    $LastChangedDate:: 2013-07-17 12:17:06 #$

        @version $LastChangedRevision: 932 $
*/

#include "source_4spinor_Local.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;

//- parameter entry
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_int_vector("source_position", std::valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Source_4spinor_Local", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Source_4spinor_Local::Parameters_Source_4spinor_Local() { append_entry(*this); }
//- end

//====================================================================
void Source_4spinor_Local::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  valarray<int> source_position;

  int err = 0;
  err += params.fetch_int_vector("source_position", source_position);

  if (err) {
    vout.crucial(m_vl, "Source_4spinor_Local: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(source_position);
}


//====================================================================
void Source_4spinor_Local::set_parameters(valarray<int>& source_position)
{
  int Ndim = CommonParameters::Ndim();

  valarray<int> Lsize(Ndim);
  Lsize[0] = CommonParameters::Lx();
  Lsize[1] = CommonParameters::Ly();
  Lsize[2] = CommonParameters::Lz();
  Lsize[3] = CommonParameters::Lt();

  //- print input parameters
  vout.general(m_vl, "Source for 4-spinor field - local:\n");
  for (int mu = 0; mu < Ndim; ++mu) {
    vout.general(m_vl, "  source_position[%d] = %d\n",
                 mu, source_position[mu]);
  }

  //- range check
  int err = 0;
  for (int mu = 0; mu < Ndim; ++mu) {
    // NB. Lsize[mu] > abs(source_position[mu])
    err += ParameterCheck::non_negative(Lsize[mu] - abs(source_position[mu]));
  }

  if (err) {
    vout.crucial(m_vl, "Source_4spinor_Local: parameter range check failed.\n");
    abort();
  }

  assert(source_position.size() == Ndim);

  //- store values
  m_source_position.resize(Ndim);
  for (int mu = 0; mu < Ndim; ++mu) {
    m_source_position[mu] = (source_position[mu] + Lsize[mu]) % Lsize[mu];
  }
}


//====================================================================
void Source_4spinor_Local::set(Field_F& src, int ic, int id)
{
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();

  valarray<int> Nsz(Ndim);
  Nsz[0] = CommonParameters::Nx();
  Nsz[1] = CommonParameters::Ny();
  Nsz[2] = CommonParameters::Nz();
  Nsz[3] = CommonParameters::Nt();

  assert(ic < CommonParameters::Nc());
  assert(id < CommonParameters::Nd());
  assert(src.nvol() == Nvol);
  assert(src.nex() == 1);

  src = 0.0;

  valarray<int> site_src(4);
  valarray<int> node_src(4);
  for (int mu = 0; mu < Ndim; ++mu) {
    site_src[mu] = m_source_position[mu] % Nsz[mu];
    node_src[mu] = m_source_position[mu] / Nsz[mu];
  }

  if (node_src[0] == Communicator::ipe(0)) {
    if (node_src[1] == Communicator::ipe(1)) {
      if (node_src[2] == Communicator::ipe(2)) {
        if (node_src[3] == Communicator::ipe(3)) {
          int site = m_index.site(site_src[0], site_src[1],
                                  site_src[2], site_src[3]);
          src.set_ri(ic, id, site, 0, 1.0, 0.0);
        }
      }
    }
  }
}


//====================================================================
//============================================================END=====
