/*!
        @file    $Id:: source_Wall.cpp #$

        @brief

        @author  <Yusuke Namekawa> namekawa@ccs.tsukuba.ac.jp(namekawa)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "source_Wall.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;

#ifdef USE_FACTORY
namespace {
  Source *create_object()
  {
    return new Source_Wall();
  }


  bool init = Source::Factory::Register("Wall", create_object);
}
#endif

//- parameter entry
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_int_vector("source_position", valarray<int>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Source.Wall", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Source_Wall::Parameters_Source_Wall() { append_entry(*this); }
//- end

//====================================================================
void Source_Wall::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  valarray<int> source_position;

  int err = 0;
  err += params.fetch_int_vector("source_position", source_position);

  if (err) {
    vout.crucial(m_vl, "Source_Wall: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(source_position);
}


//====================================================================
void Source_Wall::set_parameters(const valarray<int>& source_position)
{
  // ####  parameter setup  ####
  int Ndim = CommonParameters::Ndim();

  //- global lattice size
  valarray<int> Lsize(Ndim);
  Lsize[0] = CommonParameters::Lx();
  Lsize[1] = CommonParameters::Ly();
  Lsize[2] = CommonParameters::Lz();
  Lsize[3] = CommonParameters::Lt();

  //- local size
  valarray<int> Nsize(Ndim);
  Nsize[0] = CommonParameters::Nx();
  Nsize[1] = CommonParameters::Ny();
  Nsize[2] = CommonParameters::Nz();
  Nsize[3] = CommonParameters::Nt();

  const int t_dir = Ndim - 1;

  //- print input parameters
  vout.general(m_vl, "Source for spinor field - Wall smeared:\n");
  vout.general(m_vl, "  source_position[t] = %d\n", source_position[t_dir]);

  //- range check
  int err = 0;
  // NB. Lsize[t_dir] > abs(source_position[t_dir])
  err += ParameterCheck::non_negative(Lsize[t_dir] - abs(source_position[t_dir]));

  if (err) {
    vout.crucial(m_vl, "Source_Staggered_Wall: parameter range check failed.\n");
    abort();
  }

  //- store values
  m_source_position.resize(Ndim);
  for (int mu = 0; mu < Ndim; ++mu) {
    m_source_position[mu] = 0;
  }
  m_source_position[t_dir] = (source_position[t_dir] + Lsize[t_dir]) % Lsize[t_dir];


  //- post-process

  //- PE location in t-direction.
  int tpe = m_source_position[t_dir] / Nsize[t_dir];

  m_in_node = false;

  if (tpe == Communicator::ipe(t_dir)) {
    m_in_node = true;
  }
}


//====================================================================
void Source_Wall::set(Field& src, int j)
{
  int Ndim = CommonParameters::Ndim();

  //- global lattice size
  int Lx = CommonParameters::Lx();
  int Ly = CommonParameters::Ly();
  int Lz = CommonParameters::Lz();

  const int Lvol3 = Lx * Ly * Lz;

  //- local size
  valarray<int> Nsize(Ndim);
  Nsize[0] = CommonParameters::Nx();
  Nsize[1] = CommonParameters::Ny();
  Nsize[2] = CommonParameters::Nz();
  Nsize[3] = CommonParameters::Nt();

  //- clear field
  src = 0.0;

  if (m_in_node) {
    int t = m_source_position[3] % Nsize[3];

    for (int z = 0; z < Nsize[2]; ++z) {
      for (int y = 0; y < Nsize[1]; ++y) {
        for (int x = 0; x < Nsize[0]; ++x) {
          int isite = m_index.site(x, y, z, t);

          //XXX field layout: complex as two doubles
          src.set(2 * j, isite, 0, 1.0 / Lvol3);
        }
      }
    }
  }
}


//====================================================================
//============================================================END=====
