/*!
        @file    $Id:: source_Wall_SF.cpp #$

        @brief

        @author  <Yusuke Taniguchi> tanigchi@het.ph.tsukuba.ac.jp
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "source_Wall_SF.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;

//- parameter entry
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("ct_tilde", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Source_Wall_SF", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Source_Wall_SF::Parameters_Source_Wall_SF() { append_entry(*this); }
//- end

//====================================================================
void Source_Wall_SF::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double ct_tilde;

  int err = 0;
  err += params.fetch_double("ct_tilde", ct_tilde);

  if (err) {
    vout.crucial(m_vl, "Source_Wall_SF: fetch error, input parameter not found.\n");
    abort();
  }

  set_parameters(ct_tilde);
}


//====================================================================
void Source_Wall_SF::set_parameters(double ct_tilde)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Source_Wall_SF:\n");
  vout.general(m_vl, "  ct_tilde = %10.6f\n", ct_tilde);

  //- range check
  // NB. ct_tilde=0 is allowed.

  //- store values
  m_ct_tilde = ct_tilde;
}


//====================================================================
void Source_Wall_SF::set_parameters(Field_G *U, double ct_tilde)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Source_Wall_SF:\n");
  vout.general(m_vl, "  ct_tilde = %10.6f\n", ct_tilde);

  //- range check
  // NB. ct_tilde=0 is allowed.

  //- store values
  m_U        = U;
  m_ct_tilde = ct_tilde;
}


//====================================================================

/*!
  Use the smeared link for the SF Wall source.
*/
void Source_Wall_SF::set_parameters(Field_G *U, Director_Smear *dr_smear,
                                    double ct_tilde)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Source_Wall_SF:\n");
  vout.general(m_vl, "  ct_tilde = %10.6f\n", ct_tilde);

  //- range check
  // NB. ct_tilde=0 is allowed.

  //- store values
  dr_smear->set_config(U);
  int Nsmear = dr_smear->get_Nsmear();
  m_U = (Field_G *)dr_smear->getptr_smearedConfig(Nsmear);

  m_ct_tilde = ct_tilde;
}


//====================================================================

/*!
Set the wall source for the SF boundary propagator.
\f[
b_1\left({y},\alpha,a\right)_{\beta,b}=\widetilde{c}_t
U_0^\dagger\left(\vec{y},0\right)_{ab}
\delta_{y_0,1}\left(P_+\right)_{\alpha\beta}.
\f]
<ul>
<li>u0dag is used as \f$U_0^\dagger\left(\vec{y},0\right)_{ab}\f$.
<li>a=ac, b=ic
<li>\f$\alpha=\beta=\f$id
<li>[14 Apr 2012 Y.Taniguchi]
</ul>
 */
void Source_Wall_SF::set_t0(Field_F& src, int ic, int id)
{
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();

  valarray<int> Nsize(Ndim);
  Nsize[0] = CommonParameters::Nx();
  Nsize[1] = CommonParameters::Ny();
  Nsize[2] = CommonParameters::Nz();
  Nsize[3] = CommonParameters::Nt();

  int m_Nc = CommonParameters::Nc();
  int m_Nd = CommonParameters::Nd();

  assert(ic < m_Nc);
  assert(id < m_Nd / 2);
  assert(src.nvol() == Nvol);
  assert(src.nex() == 1);

  Mat_SU_N u0dag(m_Nc);
  src = 0.0;

  if (Communicator::ipe(3) == 0) {
    int t = 1;
    for (int z = 0; z < Nsize[2]; ++z) {
      for (int y = 0; y < Nsize[1]; ++y) {
        for (int x = 0; x < Nsize[0]; ++x) {
          int site  = m_index.site(x, y, z, t);
          int site0 = m_index.site(x, y, z, t - 1);

          u0dag = m_U->mat_dag(site0, 3);

          for (int ac = 0; ac < m_Nc; ++ac) {
            src.set_ri(ac, id, site, 0, u0dag.r(ac, ic), u0dag.i(ac, ic));
          }
        }
      }
    }
    src *= m_ct_tilde;
  }
}


//====================================================================

/*!
Set the wall source for the SF boundary propagator.
\f[
b_{T-1}\left({y},\alpha,a\right)_{\beta,b}=\widetilde{c}_tU_0(\vec{y},T-1)_{ab}
\left(P_-\right)_{\alpha\beta}\delta_{y_0,T-1}
\f]
<ul>
<li>u0 is used as \f$U_0\left(\vec{y},T-a\right)_{ab}\f$.
<li>a=ac, b=ic
<li>\f$\alpha=\beta=\f$id
<li>[14 Apr 2012 Y.Taniguchi]
</ul>
 */
void Source_Wall_SF::set_tT(Field_F& src, int ic, int id)
{
  int NPEt = CommonParameters::NPEt();
  int Nvol = CommonParameters::Nvol();
  int Ndim = CommonParameters::Ndim();

  valarray<int> Nsize(Ndim);
  Nsize[0] = CommonParameters::Nx();
  Nsize[1] = CommonParameters::Ny();
  Nsize[2] = CommonParameters::Nz();
  Nsize[3] = CommonParameters::Nt();

  int m_Nc = CommonParameters::Nc();
  int m_Nd = CommonParameters::Nd();

  assert(ic < m_Nc);
  assert(id > m_Nd / 2 - 1);
  assert(id < m_Nd);
  assert(src.nvol() == Nvol);
  assert(src.nex() == 1);

  Mat_SU_N u0(m_Nc);
  src = 0.0;

  if (Communicator::ipe(3) == (NPEt - 1)) {
    int t = Nsize[3] - 1;
    for (int z = 0; z < Nsize[2]; ++z) {
      for (int y = 0; y < Nsize[1]; ++y) {
        for (int x = 0; x < Nsize[0]; ++x) {
          int site = m_index.site(x, y, z, t);
          u0 = m_U->mat(site, 3);
          for (int ac = 0; ac < m_Nc; ++ac) {
            src.set_ri(ac, id, site, 0, u0.r(ac, ic), u0.i(ac, ic));
          }
        }
      }
    }
    src *= m_ct_tilde;
  }
}


//====================================================================
//============================================================END=====
