/*!
        @file    $Id:: gaugeFixing_Coulomb.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef GAUGEFIXING_COULOMB_INCLUDED
#define GAUGEFIXING_COULOMB_INCLUDED

#include "gaugeFixing.h"

#include "index_eo.h"
#include "shiftField_eo.h"

#include "randomNumbers.h"

#include "bridgeIO.h"
using Bridge::vout;

//- parameters class
class Parameters_GaugeFixing_Coulomb : virtual public Parameters
{
 public:
  Parameters_GaugeFixing_Coulomb();
};
//- end

//! Coulomb gauge fixing.

/*
    This class fix the gauge of configuration to the Coulomb gauge.
    The implementation assumes that the dimension is 4 and the
    Coulomb gauge fixing is performed within each time slice.
    The algorithm is that developed by the Los Alamos group [see the
    implementation note].
    Overrelaxation is incorporated.
    To escape the Gribov copy, if convergence is not reached on some
    timeslices within the iterations specified by Nreset, random
    gauge transformation is performed to reset the configuration on
    that timeslice.
    This is the reason that random number generator is needed at the
    construction of this class.

    The implementation is not complete:
    - only applies to SU(3) case: because of specific implementation
      of maxTr function (Cabibbo-Marinari maximization).
    - unnecessary arithmetic operations exist for the timeslices
      on which the gauge is already fixed to good precision.
    These should be improved in the version beyond test phase.
                                        [16 Feb 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.                [14 Nov 2012 Y.Namekawa]
*/

class GaugeFixing_Coulomb : public GaugeFixing
{
 private:

  int           m_Niter;  // max iteration number
  int           m_Nnaive; // number of naive iterations
  int           m_Nmeas;  // interval of measurements
  int           m_Nreset; // Number of iteration to reset the config.
  double        m_Enorm;  // convergence criterion
  double        m_wp;     // overrelaxation parameter
  RandomNumbers *m_rand;
  Index_eo      m_index;

 public:
  GaugeFixing_Coulomb(RandomNumbers *rand)
    : GaugeFixing(), m_Niter(0), m_rand(rand) {}

  ~GaugeFixing_Coulomb() {}

  void set_parameters(const Parameters& params);
  void set_parameters(const int Niter, const int Nnaive,
                      const int Nmeas, const int Nreset,
                      const double Enorm, const double wp);

  void fix(Field_G& Ufix, const Field_G& Uorg);

  void gauge_trans_eo(Field_G& Ue, Field_G& Uo, Field_G& Geo, int Ieo);

  void set_randomGaugeTrans(std::valarray<double>& sg, Field_G& Geo);

  //! one step of gauge fixing with overrelaxation parameter wp.
  void gfix_step(std::valarray<double>& sg,
                 Field_G& Ue, Field_G& Uo, double wp);

  void calc_SG(std::valarray<double>& sg, std::valarray<double>& Fval,
               Field_G& Ue, Field_G& Uo);
  void calc_W(Field_G& Weo, Field_G& Ue, Field_G& Uo, int Ieo);
  void calc_DLT(Field_G& Weo, Field_G& Ue, Field_G& Uo, int Ieo);

  void maxTr(Field_G&, Field_G&);
  void maxTr1(Field_G&, Field_G&);
  void maxTr2(Field_G&, Field_G&);
  void maxTr3(Field_G&, Field_G&);

  void sum_global_t(std::valarray<double>& val_global,
                    std::valarray<double>& val_local);
};
#endif
