/*!
        @file    $Id:: wilsonLoop.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#ifndef WILSONLOOP_INCLUDED
#define WILSONLOOP_INCLUDED

#include "defs.h"
#include "parameters.h"
#include "field_G.h"
#include "shiftField_eo.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Wilson loop measurement.

/*!
    This class measures Wilson loops in spatial off-axis directions
    as well as on-axis for a given gauge configuration.
    The construction of off-axis loops originates from the Fortran code
    written by Takashi Umeda (1997), while details were modified.
    The calculation is performed in the following steps.
    (1) on each node, extended gauge configuration necessary to
        determine the whole Wilson loop is set.
    (2) temporal gauge fixing is performed.
    (3) spatial products of link variables are obtained, from which
        Wilson loops are constructed.
                                          [28 May 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.                  [14 Nov 2012 Y.Namekawa]
 */

//- parameters class
class Parameters_WilsonLoop : virtual public Parameters
{
 public:
  Parameters_WilsonLoop();
};
//- end

class WilsonLoop
{
 protected:
  Bridge::VerboseLevel m_vl;

 private:
  //! parameters set by user
  int m_Nspc_size;     //!< spatial size of loop
  int m_Ntmp_size;     //!< spatial size of loop
  int m_Ntype;         //!< number of measured loop-type

  //! internal data members
  int m_Ntype_max;     //!< maximum size of loop-type
  int m_Nx_ext;        //!< size of extended gauge config.
  int m_Ny_ext;        //!< size of extended gauge config.
  int m_Nz_ext;        //!< size of extended gauge config.
  int m_Nt_ext;        //!< size of extended gauge config.
  int m_Nvol_ext;      //!< volume of extended gauge config.

  typedef std::valarray<int>   unit_vec;
  std::valarray<unit_vec> m_Nunit;
  std::valarray<int>      m_Nmax;

 public:

  WilsonLoop()
    : m_vl(CommonParameters::Vlevel())
  {
    init();
  }

  virtual ~WilsonLoop() {}

  //! setting parameters.
  virtual void set_parameters(const Parameters& params);

  void set_parameters(int Nspc_size, int Ntmp_size, int Ntype);

  void set_parameter_verboselevel(const Bridge::VerboseLevel vl) { m_vl = vl; }

  //! main function to measure Wilson loops.
  double measure(Field_G& U);

  //! index for Wilson loop variable.
  int index_wloop(int i_spc, int i_tmp, int i_type)
  {
    return i_spc + m_Nspc_size * (i_tmp + m_Ntmp_size * i_type);
  }

 private:

  //! initial setup independent of parameters.
  void init();

  //! temporal gauge fixing of extended gauge field.
  double calc_wloop(Field_G& Uspc, int t_ext);

  //! redefinition of product of spatial link variables.
  void redef_Uspc(Field_G& Uspc, Field_G& Uext,
                  int j, int nu, std::valarray<int>& unit_v);

  //! setup of extended gauge field.
  void set_extfield(Field_G& Uext, Field_G& Uorg);

  //! temporal gauge fixing of extended gauge field.
  void gfix_temporal(Field_G& Uext);
};
#endif
