/*!
        @file    $Id:: parameterManager_YAML.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
 */

#include "parameterManager_YAML.h"
#include "communicator.h"
#include <iostream>
#include <sstream>
#include <stack>
#include "evalexpr.h"

using std::string;
using std::valarray;
using Bridge::vout;

//====================================================================
void ParameterManager_YAML::read_params(const std::string& params_file, Parameters *params)
{
  const int io_node = 0;  // node id for file i/o.

  int  filesize = 0;
  char *buf     = 0;

  if (Communicator::nodeid() == io_node) {
    // load and distribute
    std::ifstream fin(params_file.c_str());
    if (!fin) {
      vout.crucial(m_vl, "ParameterManager_YAML: unable to read parameter file: %s.\n", params_file.c_str());

      Communicator::abort();
    }

    fin.seekg(0, std::ios::end);
    filesize = fin.tellg();
    fin.seekg(0, std::ios::beg);

    vout.paranoiac(m_vl, "ParameterManager_YAML::read_params: filesize = %d\n", filesize);

    Communicator::broadcast(1, &filesize, io_node);

    buf = new char [filesize];

    fin.read(buf, filesize);

    Communicator::Base::broadcast(filesize, buf, io_node);
  } else {
    // receive from io_node

    Communicator::broadcast(1, &filesize, io_node);

    buf = new char [filesize];

    Communicator::Base::broadcast(filesize, buf, io_node);
  }

  std::istringstream iss(buf);
  read_params(iss, params);

  delete [] buf;
}


//====================================================================
void ParameterManager_YAML::read_params(std::istream& fin, Parameters *params_top)
{
  typedef std::pair<int, Parameters *>   Level;

  int    linesize = 256;
  char   *ch      = new char[linesize];
  string line;
  string keystr, valstr;

  std::stack<Level> level_reserve;

  Level top_level(-1, params_top);
  level_reserve.push(top_level);

  Parameters *params = params_top;

  int indent      = 0;
  int indent_prev = indent;

  while (fin.getline(ch, linesize))
  {
    indent_prev = indent;

    line   = ch;
    indent = set_key_and_value(keystr, valstr, line);
    // extract key and value from line.

    vout.paranoiac(m_vl, "\nline: \"%s\" (%u)\n", line.c_str(), line.length());
    vout.paranoiac(m_vl, " key = \"%s\"\n", keystr.c_str());
    vout.paranoiac(m_vl, " val = \"%s\"\n", valstr.c_str());
    vout.paranoiac(m_vl, " indent = %d\n", indent);
    vout.paranoiac(m_vl, " val length = %u\n", valstr.length());

    if (indent == -1) {
      indent = indent_prev;
      continue;
    }

    if (indent < indent_prev) {
      // go to upper levels until appropriate indent level found.
      while (!level_reserve.empty())
      {
        Level level = level_reserve.top();

        if (level.first < indent) {
          params = level.second;
          break;
        }

        level_reserve.pop();
      }

      vout.paranoiac(m_vl, " go back to parent parameters.\n");
    }

    if (valstr.length() == 0) {  // map key found.
      // parameter entry for next level. skip if unused.
      params = params ? params->get_Parameters(keystr) : 0;

      // preserve parameter entry and associated indent level.
      Level level(indent, params);
      level_reserve.push(level);

      vout.paranoiac(m_vl, " go to subparameters.\n");

      continue;
    }

    if (params && params->find_int(keystr)) {
      int val = atoi(valstr.c_str());
      params->set_int(keystr, val);

      vout.paranoiac(m_vl, " int entry found for key = \"%s\", value = %d\n", keystr.c_str(), val);

      continue;
    }

    if (params && params->find_double(keystr)) {
//      double val = (valstr.c_str());
      double val = EvalExpr(valstr).parse();
      params->set_double(keystr, val);

      vout.paranoiac(m_vl, " double entry found for key = \"%s\", value = %f\n", keystr.c_str(), val);

      continue;
    }

    if (params && params->find_int_vector(keystr)) {
      valarray<int> vec;
      convert_int_vector(vec, valstr);
      params->set_int_vector(keystr, vec);

      vout.paranoiac(m_vl, " int_vector entry found for key = \"%s\"\n", keystr.c_str());

      continue;
    }

    if (params && params->find_double_vector(keystr)) {
      valarray<double> vec;
      convert_double_vector(vec, valstr);
      params->set_double_vector(keystr, vec);

      vout.paranoiac(m_vl, " double_vector entry found for key = \"%s\"\n", keystr.c_str());

      continue;
    }

    if (params && params->find_string(keystr)) {
      params->set_string(keystr, valstr);

      vout.paranoiac(m_vl, " string entry found for key = \"%s\", value = \"%s\"\n", keystr.c_str(), valstr.c_str());

      continue;
    }

    // verbosity entry
    if (params && (keystr == "verbose_level")) {
      Bridge::VerboseLevel vl = Bridge::BridgeIO::set_verbose_level(valstr);
      params->set_VerboseLevel(vl);

      vout.paranoiac(m_vl, " verbosity entry found, value = %s(%d)\n", valstr.c_str(), vl);

      continue;
    }

    // vout.crucial(m_vl, "ParameterManager_YAML: entry not found.\n");
    // Communicator::abort();
  }

  delete [] ch;
}


//====================================================================
int ParameterManager_YAML::set_key_and_value(string& keystr, string& valstr, string& line)
{
  string::size_type i2 = line.find("#");

  if (i2 != string::npos) {
    line.erase(i2, line.length() - i2);
  }

  string::size_type i1 = line.find(":");
  //    cout << " position of : = " << i1 << endl;
  if (i1 == string::npos) {
    //    cout << "npos\n";
    keystr = "";
    valstr = "";
    return -1;    // no ":" in the line: regarded as a blank line
  }

  keystr = line.substr(0, i1);
  valstr = line.substr(i1 + 1, line.length() - i1 - 1);

  int indent = remove_space(keystr);
  remove_space(valstr);

  return indent;
}


//====================================================================
int ParameterManager_YAML::remove_space(string& str)
{
  int indent = 0;

  while (str.length() > 0 && str[0] == ' ')
  {
    str.erase(0, 1);
    ++indent;
  }

  while (str.length() > 0 && str[str.length() - 1] == ' ')
  {
    str.erase(str.length() - 1, 1);
  }

  return indent;
}


//====================================================================
void ParameterManager_YAML::convert_int_vector(valarray<int>& vec,
                                               string& valstr)
{
  int size_max = 20;

  valarray<int> vec_tmp(size_max);
  int           size = 0;

  //  cout << valstr << "--" << endl;

  if (valstr[0] != '[') {
    vout.crucial(m_vl, "ParameterManager_YAML: wrong format of vector.\n");
    Communicator::abort();
  }
  valstr.erase(0, 1);

  if (valstr[valstr.length() - 1] != ']') {
    vout.crucial(m_vl, "ParameterManager_YAML: wrong format of vector.\n");
    Communicator::abort();
  }
  valstr.erase(valstr.length() - 1, 1);

  //  cout << valstr << "--" << endl;


  while (valstr.length() > 0)
  {
    string::size_type i1 = valstr.find(",");
    //    cout << " position of : = " << i1 << endl;
    if (i1 != string::npos) {
      string elem = valstr.substr(0, i1);
      valstr.erase(0, i1 + 1);
      remove_space(elem);
      //  cout << elem << "--" << endl;
      vec_tmp[size] = atoi(elem.c_str());
      ++size;
      //  cout << valstr << "--"  << endl;
    } else {
      string elem = valstr;
      valstr.erase(0, elem.length());
      remove_space(elem);
      //  cout << elem << "--" << endl;
      vec_tmp[size] = atoi(elem.c_str());
      ++size;
    }

    if (size > size_max) {
      vout.crucial(m_vl, "ParameterManager_YAML: too large vector size.\n");
      Communicator::abort();
    }
  }

  vec.resize(size);
  for (int i = 0; i < size; ++i) {
    vec[i] = vec_tmp[i];
    // cout << "vec[" << i << "] = " << vec[i] << endl;
  }
}


//====================================================================
void ParameterManager_YAML::convert_double_vector(valarray<double>& vec,
                                                  string& valstr)
{
  int size_max = 20;

  valarray<double> vec_tmp(size_max);
  int              size = 0;

  //  cout << valstr << "--" << endl;

  if (valstr[0] != '[') {
    vout.crucial(m_vl, "ParameterManager_YAML: wrong format of vector.\n");
    Communicator::abort();
  }
  valstr.erase(0, 1);

  if (valstr[valstr.length() - 1] != ']') {
    vout.crucial(m_vl, "ParameterManager_YAML: wrong format of vector.\n");
    Communicator::abort();
  }
  valstr.erase(valstr.length() - 1, 1);

  //  cout << valstr << "--" << endl;


  while (valstr.length() > 0)
  {
    string::size_type i1 = valstr.find(",");
    //    cout << " position of : = " << i1 << endl;
    if (i1 != string::npos) {
      string elem = valstr.substr(0, i1);
      valstr.erase(0, i1 + 1);
      remove_space(elem);
      //  cout << elem << "--" << endl;
      // vec_tmp[size] = atof(elem.c_str());
      vec_tmp[size] = EvalExpr(elem).parse();
      ++size;
      //  cout << valstr << "--"  << endl;
    } else {
      string elem = valstr;
      valstr.erase(0, elem.length());
      remove_space(elem);
      //  cout << elem << "--" << endl;
      // vec_tmp[size] = atof(elem.c_str());
      vec_tmp[size] = EvalExpr(elem).parse();
      ++size;
    }

    if (size > size_max) {
      vout.crucial(m_vl, "ParameterManager_YAML: too large vector size.\n");
      Communicator::abort();
    }
  }

  vec.resize(size);
  for (int i = 0; i < size; ++i) {
    vec[i] = vec_tmp[i];
    //    cout << "vec[" << i << "] = " << vec[i] << endl;
  }
}


//====================================================================
//============================================================END=====
