/*!
        @file    $Id:: parameters.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
 */

#include "parameters.h"

//====================================================================
Parameters::Parameters()
  : m_vlevel(CommonParameters::Vlevel())
{
  // default verbosity from common_parameters.
}


//====================================================================
double Parameters::get_double(const string& key) const
{
  std::map<string, Item<double> >::const_iterator p = m_map_double.find(key);

  if (p != m_map_double.end()) {
    return p->second.value();
  }

  vout.crucial("get_double: key '%s' not found.\n", key.c_str());

  return double();
}


//====================================================================
int Parameters::get_int(const string& key) const
{
  std::map<string, Item<int> >::const_iterator p = m_map_int.find(key);

  if (p != m_map_int.end()) {
    return p->second.value();
  }

  vout.crucial("get_int: key '%s' not found.\n", key.c_str());

  return int();
}


//====================================================================
std::valarray<double> Parameters::get_double_vector(const string& key) const
{
  std::map<string, Item<std::valarray<double> > >::const_iterator p = m_map_double_vector.find(key);

  if (p != m_map_double_vector.end()) {
    return p->second.value();
  }

  vout.crucial("get_double_vector: key '%s' not found.\n", key.c_str());

  return std::valarray<double>();
}


//====================================================================
std::valarray<int> Parameters::get_int_vector(const string& key) const
{
  std::map<string, Item<std::valarray<int> > >::const_iterator p = m_map_int_vector.find(key);

  if (p != m_map_int_vector.end()) {
    return p->second.value();
  }

  vout.crucial("get_int_vector: key '%s' not found.\n", key.c_str());

  return std::valarray<int>();
}


//====================================================================
string Parameters::get_string(const string& key) const
{
  std::map<string, Item<string> >::const_iterator p = m_map_string.find(key);

  if (p != m_map_string.end()) {
    return p->second.value();
  }

  vout.crucial("get_Parameters: key '%s' not found.\n", key.c_str());

  return string();
}


//====================================================================
Parameters *Parameters::get_Parameters(const string& key) const
{
  std::map<string, Item<Parameters *> >::const_iterator p = m_map_Parameters.find(key);

  if (p != m_map_Parameters.end()) {
    return p->second.value();
  }

  vout.crucial("Warning: get_Parameters: key '%s' is not used.\n", key.c_str());

  return 0;
}


//====================================================================
// add by SU
Bridge::VerboseLevel Parameters::get_VerboseLevel() const
{
  return m_vlevel.value();
}


//====================================================================
// bool Parameters::fetch_double(const string& key, double& val) const
int Parameters::fetch_double(const string& key, double& val) const
{
  std::map<string, Item<double> >::const_iterator p = m_map_double.find(key);

  if (p == m_map_double.end()) {
    vout.crucial("Parameters: parameter entry not found: %s.\n", key.c_str());
    abort();
    return EXIT_FAILURE;
  }

  val = p->second.value();

  return p->second.is_set();
}


//====================================================================
int Parameters::fetch_int(const string& key, int& val) const
{
  std::map<string, Item<int> >::const_iterator p = m_map_int.find(key);

  if (p == m_map_int.end()) {
    vout.crucial("Parameters: parameter entry not found: %s.\n", key.c_str());
    abort();
    return EXIT_FAILURE;
  }

  val = p->second.value();

  return p->second.is_set();
}


//====================================================================
int Parameters::fetch_double_vector(const string& key, std::valarray<double>& val) const
{
  std::map<string, Item<std::valarray<double> > >::const_iterator p = m_map_double_vector.find(key);

  if (p == m_map_double_vector.end()) {
    vout.crucial("Parameters: parameter entry not found: %s.\n", key.c_str());
    abort();
    return EXIT_FAILURE;
  }

  val.resize(p->second.value().size());
  val = p->second.value();

  return p->second.is_set();
}


//====================================================================
int Parameters::fetch_int_vector(const string& key, std::valarray<int>& val) const
{
  std::map<string, Item<std::valarray<int> > >::const_iterator p = m_map_int_vector.find(key);

  if (p == m_map_int_vector.end()) {
    vout.crucial("Parameters: parameter entry not found: %s.\n", key.c_str());
    abort();
    return EXIT_FAILURE;
  }

  val.resize(p->second.value().size());
  val = p->second.value();

  return p->second.is_set();
}


//====================================================================
int Parameters::fetch_string(const string& key, string& val) const
{
  std::map<string, Item<string> >::const_iterator p = m_map_string.find(key);

  if (p == m_map_string.end()) {
    vout.crucial("Parameters: parameter entry not found: %s.\n", key.c_str());
    abort();
    return EXIT_FAILURE;
  }

  val = p->second.value();

  return p->second.is_set();
}


//====================================================================
int Parameters::fetch_VerboseLevel(Bridge::VerboseLevel& val) const
{
  val = m_vlevel.value();

  return m_vlevel.is_set();
}


//====================================================================
bool Parameters::find_int(const string& key) const
{
  return m_map_int.find(key) != m_map_int.end();
}


//====================================================================
bool Parameters::find_double(const string& key) const
{
  return m_map_double.find(key) != m_map_double.end();
}


//====================================================================
bool Parameters::find_int_vector(const string& key) const
{
  return m_map_int_vector.find(key) != m_map_int_vector.end();
}


//====================================================================
bool Parameters::find_double_vector(const string& key) const
{
  return m_map_double_vector.find(key) != m_map_double_vector.end();
}


//====================================================================
bool Parameters::find_string(const string& key) const
{
  return m_map_string.find(key) != m_map_string.end();
}


//====================================================================
bool Parameters::find_Parameters(const string& key) const
{
  return m_map_Parameters.find(key) != m_map_Parameters.end();
}


//====================================================================
void Parameters::set_int(const string& key, const int value)
{
  std::map<string, Item<int> >::iterator p = m_map_int.find(key);

  if (p != m_map_int.end()) p->second.put(value);
}


//====================================================================
void Parameters::set_double(const string& key, const double value)
{
  std::map<string, Item<double> >::iterator p = m_map_double.find(key);

  if (p != m_map_double.end()) p->second.put(value);
}


//====================================================================
void Parameters::set_int_vector(const string& key, const std::valarray<int>& value)
{
  std::map<string, Item<std::valarray<int> > >::iterator p = m_map_int_vector.find(key);

  if (p != m_map_int_vector.end()) p->second.put(value);
}


//====================================================================
void Parameters::set_double_vector(const string& key, const std::valarray<double>& value)
{
  std::map<string, Item<std::valarray<double> > >::iterator p = m_map_double_vector.find(key);

  if (p != m_map_double_vector.end()) p->second.put(value);
}


//====================================================================
void Parameters::set_string(const string& key, const string& value)
{
  std::map<string, Item<string> >::iterator p = m_map_string.find(key);

  if (p != m_map_string.end()) p->second.put(value);
}


//====================================================================
void Parameters::set_Parameters(const string& key, Parameters *const value)
{
  std::map<string, Item<Parameters *> >::iterator p = m_map_Parameters.find(key);

  if (p != m_map_Parameters.end()) p->second.put(value);
}


//====================================================================
// add by SU
void Parameters::set_VerboseLevel(Bridge::VerboseLevel value)
{
  m_vlevel.put(value);
}


//====================================================================
void Parameters::Register_double(const string& key, const double data)
{
  m_map_double.insert(std::map<string, Item<double> >::value_type(key, Item<double>(data)));
}


//====================================================================
void Parameters::Register_int(const string& key, const int data)
{
  m_map_int.insert(std::map<string, Item<int> >::value_type(key, Item<int>(data)));
}


//====================================================================
void Parameters::Register_double_vector(const string& key, const std::valarray<double>& data)
{
  m_map_double_vector.insert(std::map<string, Item<std::valarray<double> > >::value_type(key, Item<std::valarray<double> >(data)));
}


//====================================================================
void Parameters::Register_int_vector(const string& key, const std::valarray<int>& data)
{
  m_map_int_vector.insert(std::map<string, Item<std::valarray<int> > >::value_type(key, Item<std::valarray<int> >(data)));
}


//====================================================================
void Parameters::Register_string(const string& key, const string& data)
{
  m_map_string.insert(std::map<string, Item<string> >::value_type(key, Item<string>(data)));
}


//====================================================================
void Parameters::Register_Parameters(const string& key, Parameters *const data)
{
  m_map_Parameters.insert(std::map<string, Item<Parameters *> >::value_type(key, Item<Parameters *>(data)));
}


#ifdef DEBUG
//====================================================================
void Parameters::dump(const string& indent_str) const
{
  const char *indent = indent_str.c_str();

  if (m_map_double.size() > 0) {
    printf("%s[double]\n", indent);
    for (std::map<string, Item<double> >::const_iterator p = m_map_double.begin(); p != m_map_double.end(); ++p) {
      printf("%s  key = %s, val = %f, set = %s\n", indent, p->first.c_str(), p->second.value(), (p->second.m_flag_set ? "true" : "false"));
    }
  }

  if (m_map_int.size() > 0) {
    printf("%s[integer]\n", indent);
    for (std::map<string, Item<int> >::const_iterator p = m_map_int.begin(); p != m_map_int.end(); ++p) {
      printf("%s  key = %s, val = %d, set = %s\n", indent, p->first.c_str(), p->second.value(), (p->second.m_flag_set ? "true" : "false"));
    }
  }

  if (m_map_double_vector.size() > 0) {
    printf("%s[double vector]\n", indent);
    for (std::map<string, Item<std::valarray<double> > >::const_iterator p = m_map_double_vector.begin(); p != m_map_double_vector.end(); ++p) {
      printf("%s  key = %s, val = (", indent, p->first.c_str());
      for (size_t i = 0; i < p->second.value().size(); ++i) {
        printf("%f, ", (p->second.value())[i]);
      }
      printf("), set = %s\n", (p->second.m_flag_set ? "true" : "false"));
    }
  }

  if (m_map_int_vector.size() > 0) {
    printf("%s[integer vector]\n", indent);
    for (std::map<string, Item<std::valarray<int> > >::const_iterator p = m_map_int_vector.begin(); p != m_map_int_vector.end(); ++p) {
      printf("%s  key = %s, val = (", indent, p->first.c_str());
      for (size_t i = 0; i < p->second.value().size(); ++i) {
        printf("%d, ", (p->second.value())[i]);
      }
      printf("), set = %s\n", (p->second.m_flag_set ? "true" : "false"));
    }
  }

  if (m_map_string.size() > 0) {
    printf("%s[string]\n", indent);
    for (std::map<string, Item<string> >::const_iterator p = m_map_string.begin(); p != m_map_string.end(); ++p) {
      printf("%s  key = %s, val = %s, set = %s\n", indent, p->first.c_str(), p->second.value().c_str(), (p->second.m_flag_set ? "true" : "false"));
    }
  }

  if (m_map_Parameters.size() > 0) {
    printf("%s[Parameters]\n", indent);
    for (std::map<string, Item<Parameters *> >::const_iterator p = m_map_Parameters.begin(); p != m_map_Parameters.end(); ++p) {
      printf("%s  key = %s, val = %p, set = %s\n", indent, p->first.c_str(), p->second.value(), (p->second.m_flag_set ? "true" : "false"));

      // dump recursively.
      if (p->second.value()) {
        p->second.value()->dump(indent_str + "  ");
      } else {
        printf("%s<empty value>\n", indent);
      }
    }
  }

  printf("%s[verbosity]\n", indent);
  printf("%s  value = %d, set = %s\n", indent, m_vlevel.value(), (m_vlevel.m_flag_set ? "true" : "false"));

  printf("%s--------\n", indent);
}
#endif

//====================================================================
//============================================================END=====
