/*!
        @file    $Id:: parameters.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
 */

#ifndef PARAMETERS_INCLUDED
#define PARAMETERS_INCLUDED

#include <string>
using std::string;

#include <map>
#include <valarray>

#include "commonParameters.h"
#include "communicator.h"
#include "checker.h"

#include "bridgeIO.h"
using Bridge::vout;

//! Class for parameters

/*!
    Base class of Parameters.
    This class gives general basis of structured parameter sets.
                                       [17 Jul 2012 H.Matsufuru]
    fetch is modified to return int, instead of bool in the original
    Aoyama-san's code.                 [16 Jun 2013 Y.Namekawa]
 */

class Parameters
{
 private:

  template<typename T>
  struct Item
  {
    Item(const T& value)
      : m_value(value), m_flag_set(EXIT_FAILURE) {}
    //: m_value(value), m_flag_set(false) {}

    T        m_value;
    // bool m_flag_set;
    int      m_flag_set;

    const T& value() const { return m_value; }
    // bool is_set() const { return m_flag_set; }
    int      is_set() const { return m_flag_set; }

    template<typename U>
    void put(const std::valarray<U>& value)
    {
      m_value.resize(value.size());
      m_value = value;
      // m_flag_set = true;
      m_flag_set = EXIT_SUCCESS;
    }

    template<typename U>
    void put(const U& value)
    {
      m_value = value;
      // m_flag_set = true;
      m_flag_set = EXIT_SUCCESS;
    }
  };

  std::map<string, Item<double> >                 m_map_double;
  std::map<string, Item<int> >                    m_map_int;
  std::map<string, Item<std::valarray<double> > > m_map_double_vector;
  std::map<string, Item<std::valarray<int> > >    m_map_int_vector;
  std::map<string, Item<string> >                 m_map_string;
  std::map<string, Item<Parameters *> >           m_map_Parameters;
  Item<Bridge::VerboseLevel> m_vlevel;

 public:

  Parameters();

  virtual ~Parameters() {}

  void set_double(const string& key, const double value);
  void set_int(const string& key, const int value);
  void set_double_vector(const string& key, const std::valarray<double>& value);
  void set_int_vector(const string& key, const std::valarray<int>& value);
  void set_string(const string& key, const string& value);
  void set_Parameters(const string& key, Parameters *const value);
  void set_VerboseLevel(Bridge::VerboseLevel value);

  double get_double(const string& key) const;
  int get_int(const string& key) const;

  std::valarray<double> get_double_vector(const string& key) const;

  std::valarray<int> get_int_vector(const string& key) const;
  string get_string(const string& key) const;
  Parameters *get_Parameters(const string& key) const;
  Bridge::VerboseLevel get_VerboseLevel() const;

  // bool fetch_double(       const string& key, double& val) const;
  int fetch_double(const string& key, double& val) const;
  int fetch_int(const string& key, int& val) const;
  int fetch_double_vector(const string& key, std::valarray<double>& val) const;
  int fetch_int_vector(const string& key, std::valarray<int>& val) const;
  int fetch_string(const string& key, string& val) const;
  int fetch_VerboseLevel(Bridge::VerboseLevel& val) const;

  void Register_double(const string&, const double);
  void Register_int(const string&, const int);
  void Register_double_vector(const string&, const std::valarray<double>&);
  void Register_int_vector(const string&, const std::valarray<int>&);
  void Register_string(const string&, const string&);
  void Register_Parameters(const string&, Parameters *const);

  bool find_double(const string&) const;
  bool find_int(const string&) const;
  bool find_double_vector(const string&) const;
  bool find_int_vector(const string&) const;
  bool find_string(const string&) const;
  bool find_Parameters(const string&) const;

#ifdef DEBUG
  void dump(const string& indent = "") const;
#endif
};
#endif
