/*!
        @file    $Id:: parameters_fa#$

        @brief

        @author  $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-22 16:55:30 #$

        @version $LastChangedRevision: 943 $
*/

#ifndef PARAMETERS_FACTORY_INCLUDED
#define PARAMETERS_FACTORY_INCLUDED

#include <map>
#include <list>
#include <vector>
#include <string>
#include "defs.h"
#include "parameters.h"
#include "bridgeIO.h"

//! Factory for Parameters class instance

/**
   ParametersFactory is a factory for Parameters class instances.

   Many classes are controlled by external parameters that are packed
   in a form of Parameters class. ParametersFactory provides a way to
   generate an instance of Parameters associated to the class through
   keyword representing the class hierarchy.
   It is so designed that a generic Parameters that applies to a family
   of classes (e.g. Solver) can be generated as well as the individual
   concrete class (e.g. Solver.CG for CG solver class).

   The form of the keyword (realm) is a concatenation of class tree
   by '.' (dot). An example is shown above.

   An instance of Parameters class is obtained through calling class
   method ParametersFactory::New() with the class keyword. it should
   be deleted later by uses.

   A callback function to generate an instance is registered through
   ParametersFactory::Register(). The callback function takes a reference
   to Parameters, appends parameter entries to it, and returns.

   By construction, ParametersClass is a singleton.
*/

class Parameters;

class ParametersFactory {
 private:
  typedef void (*creator_callback)(Parameters&);
  typedef std::list<creator_callback>                    creator_callback_list;
  typedef std::map<std::string, creator_callback_list>   database_type;

 public:
  Parameters *create_parameters(const std::string& realm);

  bool register_callback(const std::string& realm, const creator_callback& cb);

  static ParametersFactory& Instance();

  static Parameters *New(const std::string& realm)
  {
    return Instance().create_parameters(realm);
  }

  static bool Register(const std::string& realm, const creator_callback& cb)
  {
    return Instance().register_callback(realm, cb);
  }

 private:
  // singleton
  ParametersFactory() {}
  ParametersFactory(const ParametersFactory&);            // left unimplemented.
  ParametersFactory& operator=(const ParametersFactory&); // left unimplemented.

  ~ParametersFactory() {}

  static void CreateInstance();

  static ParametersFactory *s_instance;

  std::vector<std::string> string_tokenize(const std::string& src, const char delim = '.');
  std::string string_concat(const std::vector<std::string>& src, const char delim = '.');

  database_type m_map;
};
#endif
