/*!
        @file    $Id:: smear_APE.h #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/


#ifndef SMEAR_APE_INCLUDED
#define SMEAR_APE_INCLUDED

#include <valarray>
#include "smear.h"
#include "staples.h"

#include "bridgeIO.h"
using Bridge::vout;

//- parameters class
//! Parameters class for APE type link smearing.
class Parameters_Smear_APE : virtual public Parameters
{
 public:
  Parameters_Smear_APE();
};
//- end

//! APE type smearing of link variables.

/*!
    This class smears link variables with APE-type construction
    of smeared links with a given projection operator to SU(N)
    group element.
    Parameter is \rho(\mu,\nu), which in general depends on
    the directions of the modified link and the staple.
    By explicitly giving \rho(\mu,\nu) as valarray object,
    anisotropic setup is possible, while isotropic setup
    requires only one double parameter, `rho_uniform'.
                       [08 Apr 2012/15 Jul 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.            [14 Nov 2012 Y.Namekawa]
 */

class Smear_APE : public Smear
{
 private:
  int m_Ndim;                    //!< spacetime dimension
  std::valarray<double> m_rho;   //!< smearing parameter
  Projection            *m_proj; //!< projector to group element.

 public:
  //! Constructor requires a pointer to Projection object.
  Smear_APE(Projection *proj)
    : Smear(),
      m_Ndim(CommonParameters::Ndim()), m_rho(0.0, m_Ndim * m_Ndim),
      m_proj(proj) {}

  //! Deconstructor
  ~Smear_APE() {}

  //! Setting parameters with Parameters object.
  void set_parameters(const Parameters& params);

  //! Setting parameter with isotropic parameter.
  void set_parameters(const double rho1);

  //! Setting parameter with anisotropic parameter.
  void set_parameters(const std::valarray<double>& rho);

  //! Smearing of a given gauge field.
  void smear(Field_G& Usmear, const Field_G& U);

 private:
  //! Staple construction.
  void staple(Field_G&, const Field_G&, const Field_G&,
              int mu, int nu);
};
#endif
