/*!
        @file    $Id:: smear_APE_spatial.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-12 16:56:41 #$

        @version $LastChangedRevision: 930 $
*/

#include "smear_APE_spatial.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

#ifdef USE_FACTORY
namespace {
  Smear *create_object(Projection *proj)
  {
    return new Smear_APE_spatial(proj);
  }


  bool init = Smear::Factory::Register("APE_spatial", create_object);
}
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("rho", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Smear.APE_Spatial", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Smear_APE_spatial::Parameters_Smear_APE_spatial() { append_entry(*this); }
//- end

//====================================================================
void Smear_APE_spatial::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double rho;

  int err = 0;
  err += params.fetch_double("rho", rho);

  if (err) {
    vout.crucial(m_vl, "Smear_APE_Spatial: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(rho);
}


//====================================================================
void Smear_APE_spatial::set_parameters(const double rho)
{
  //- print input parameters
  vout.general(m_vl, "APE spatial link smearing parameters:\n");
  vout.general(m_vl, "  rho = %10.6F\n", rho);

  //- range check
  // NB. rho == 0 is allowed.

  //- store values
  m_rho = rho;
}


//====================================================================
void Smear_APE_spatial::smear(Field_G& Usmear, const Field_G& U)
{
  int Nvol = CommonParameters::Nvol();

  assert(U.nvol() == Nvol);
  assert(U.nex() == m_Ndim);

  assert(Usmear.nvol() == Nvol);
  assert(Usmear.nex() == m_Ndim);

  int Ndim_spc = m_Ndim - 1;

  Field_G c_tmp(Nvol, 1), u_tmp(Nvol, 1), u_tmp2(Nvol, 1);

  Staples staple;

  double plaq = staple.plaq_s(U);
  vout.general(m_vl, "  plaq_s(org  ) = %12.8f\n", plaq);
  plaq = staple.plaq_t(U);
  vout.general(m_vl, "  plaq_t(org  ) = %12.8f\n", plaq);

  Usmear = 0.0;

  for (int mu = 0; mu < Ndim_spc; ++mu) {
    c_tmp = 0.0;
    u_tmp.setpart_ex(0, U, mu);

    for (int nu = 0; nu < Ndim_spc; ++nu) {
      if (nu != mu) {
        u_tmp2 = staple.upper(U, mu, nu);
        c_tmp.addpart_ex(0, u_tmp2, 0, m_rho);

        u_tmp2 = staple.lower(U, mu, nu);
        c_tmp.addpart_ex(0, u_tmp2, 0, m_rho);
      }
    }

    m_proj->project(u_tmp2, m_rho, c_tmp, u_tmp);
    Usmear.setpart_ex(mu, u_tmp2, 0);
  }

  int mu = m_Ndim - 1; // temporal link: unsmeared.
  Usmear.setpart_ex(mu, U, mu);

  plaq = staple.plaq_s(Usmear);
  vout.general(m_vl, "  plaq_s(smear) = %12.8f\n", plaq);
  plaq = staple.plaq_t(Usmear);
  vout.general(m_vl, "  plaq_t(smear) = %12.8f\n", plaq);
}


//====================================================================
//============================================================END=====
