/*!
        @file    $Id:: smear_HYP_SF.cpp #$

        @brief

        @author  Yusuke Tanigchi<tanigchi@ht.ph.tsukuba.ac.jp> (tanigchi)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "smear_HYP_SF.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

using std::valarray;

#ifdef USE_FACTORY
namespace {
  Smear *create_object(Projection *proj)
  {
    return new Smear_HYP_SF(proj);
  }


  bool init = Smear::Factory::Register("HYP_SF", create_object);
}
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_double("alpha1", 0.0);
    param.Register_double("alpha2", 0.0);
    param.Register_double("alpha3", 0.0);
    param.Register_double_vector("phi", std::valarray<double>());
    param.Register_double_vector("phipr", std::valarray<double>());

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Smear.HYP_SF", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Smear_HYP_SF::Parameters_Smear_HYP_SF() { append_entry(*this); }
//- end

//====================================================================
void Smear_HYP_SF::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  double           alpha1, alpha2, alpha3;
  valarray<double> phi, phipr;

  int err = 0;
  err += params.fetch_double("alpha1", alpha1);
  err += params.fetch_double("alpha2", alpha2);
  err += params.fetch_double("alpha3", alpha3);
  err += params.fetch_double_vector("phi", phi);
  err += params.fetch_double_vector("phipr", phipr);

  if (err) {
    vout.crucial(m_vl, "Smear_HYP_SF: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(alpha1, alpha2, alpha3, &phi[0], &phipr[0]);
}


//====================================================================
void Smear_HYP_SF::set_parameters(double alpha1, double alpha2, double alpha3,
                                  double *phi, double *phipr)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Smear_HYP_SF:\n");
  vout.general(m_vl, "  alpha1 = %10.6F\n", alpha1);
  vout.general(m_vl, "  alpha2 = %10.6F\n", alpha2);
  vout.general(m_vl, "  alpha3 = %10.6F\n", alpha3);

  vout.general(m_vl, "  phi1  = %12.6f\n", phi[0]);
  vout.general(m_vl, "  phi2  = %12.6f\n", phi[1]);
  vout.general(m_vl, "  phi3  = %12.6f\n", phi[2]);
  vout.general(m_vl, "  phipr1= %12.6f\n", phipr[0]);
  vout.general(m_vl, "  phipr2= %12.6f\n", phipr[1]);
  vout.general(m_vl, "  phipr3= %12.6f\n", phipr[2]);

  //- range check
  // NB. alpha1,alpha2,alpha3 == 0 is allowed.
  // NB. phi,phipr == 0 is allowed.

  //- store values
  m_alpha1 = alpha1;
  m_alpha2 = alpha2;
  m_alpha3 = alpha3;

  //- post-process
  set_wk.set_parameters(phi, phipr);
}


//====================================================================
void Smear_HYP_SF::init()
{
  m_Ndim = CommonParameters::Ndim();
  m_Nvol = CommonParameters::Nvol();

  m_U.resize(m_Ndim);
  m_v1.resize(size_v1());
  m_v2.resize(size_v2());
}


//====================================================================
void Smear_HYP_SF::smear(Field_G& Usmear, const Field_G& U)
{
  assert(U.nvol() == m_Nvol);
  assert(U.nex() == m_Ndim);

  assert(Usmear.nvol() == m_Nvol);
  assert(Usmear.nex() == m_Ndim);

  for (int mu = 0; mu < m_Ndim; ++mu) {
    m_U[mu].setpart_ex(0, U, mu);
    if (mu != 3) set_wk.set_boundary_wk(m_U[mu]);
  }

  step1();
  //  vout.general(m_vl,"level-1 step finished.\n");
  step2();
  //  vout.general(m_vl,"level-2 step finished.\n");
  step3(Usmear);
  //  vout.general(m_vl,"level-3 step finished.\n");
}


//====================================================================
void Smear_HYP_SF::step1()
{
  Field_G c1(m_Nvol, 1);

  for (int mu = 0; mu < m_Ndim; ++mu) {
    for (int nu = 0; nu < m_Ndim; ++nu) {
      if (nu == mu) continue;

      for (int rho = nu + 1; rho < m_Ndim; ++rho) {
        if (rho == mu) continue;
        int sig = 6 - mu - nu - rho;
        staple(c1, m_U[mu], m_U[sig], mu, sig);
        c1 *= m_alpha3 / 2.0;
        m_proj->project(m_v1[index_v1(mu, nu, rho)], m_alpha3, c1, m_U[mu]);

        if (mu != 3) set_wk.set_boundary_wk(m_v1[index_v1(mu, nu, rho)]);
      }
    }
  }
}


//====================================================================
void Smear_HYP_SF::step2()
{
  Field_G c2(m_Nvol, 1), u_tmp(m_Nvol, 1);

  for (int mu = 0; mu < m_Ndim; ++mu) {
    for (int nu = 0; nu < m_Ndim; ++nu) {
      if (nu == mu) continue;
      c2 = 0.0;

      for (int rho = 0; rho < m_Ndim; ++rho) {
        if ((rho != mu) && (rho != nu)) {
          staple(u_tmp, m_v1[index_v1(mu, nu, rho)],
                 m_v1[index_v1(rho, nu, mu)], mu, rho);
          c2.addpart_ex(0, u_tmp, 0);
        }
      }

      c2 *= m_alpha2 / 4.0;
      m_proj->project(m_v2[index_v2(mu, nu)], m_alpha2, c2, m_U[mu]);

      if (mu != 3) set_wk.set_boundary_wk(m_v2[index_v2(mu, nu)]);
    }
  }
}


//====================================================================
void Smear_HYP_SF::step3(Field_G& v)
{
  Field_G c3(m_Nvol, 1), u_tmp(m_Nvol, 1);

  for (int mu = 0; mu < m_Ndim; ++mu) {
    c3 = 0.0;

    for (int nu = 0; nu < m_Ndim; ++nu) {
      if (nu != mu) {
        staple(u_tmp, m_v2[index_v2(mu, nu)],
               m_v2[index_v2(nu, mu)], mu, nu);
        c3.addpart_ex(0, u_tmp, 0);
      }
    }

    c3 *= m_alpha1 / 6.0;
    m_proj->project(u_tmp, m_alpha1, c3, m_U[mu]);

    if (mu != 3) set_wk.set_boundary_wk(u_tmp);

    v.setpart_ex(mu, u_tmp, 0);
  }
}


//====================================================================
void Smear_HYP_SF::staple(Field_G& c,
                          const Field_G& u_mu, const Field_G& u_nu,
                          int mu, int nu)
{
  Field_G v1(m_Nvol, 1), v2(m_Nvol, 1);

  //- upper direction
  m_shift.backward(v1, u_mu, nu);
  if (nu == 3) set_wk.set_boundary_wkpr(v1);
  v2.mult_Field_Gnn(0, u_nu, 0, v1, 0);

  m_shift.backward(v1, u_nu, mu);
  if (mu == 3) set_wk.set_boundary_wkpr(v1);
  c.mult_Field_Gnd(0, v2, 0, v1, 0);

  //- lower direction
  m_shift.backward(v2, u_nu, mu);
  if (mu == 3) set_wk.set_boundary_wkpr(v2);
  v1.mult_Field_Gnn(0, u_mu, 0, v2, 0);
  v2.mult_Field_Gdn(0, u_nu, 0, v1, 0);
  m_shift.forward(v1, v2, nu);
  c.addpart_ex(0, v1, 0);
  //  if(mu!=3) c.set_boundary_zero();
}


//====================================================================
//============================================================END=====
