/*!
        @file    $Id:: solver_BiCGStab_Cmplx.cpp #$

        @brief

        @author  <Yusuke Namekawa> namekawa@ccs.tsukuba.ac.jp(namekawa)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "solver_BiCGStab_Cmplx.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

#ifdef USE_FACTORY
namespace {
  Solver *create_object(Fopr *fopr)
  {
    return new Solver_BiCGStab_Cmplx(fopr);
  }


  bool init = Solver::Factory::Register("BiCGStab_Cmplx", create_object);
}
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_int("maximum_number_of_iteration", 0);
    param.Register_double("convergence_criterion_squared", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Solver.BiCGStab_Cmplx", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Solver_BiCGStab_Cmplx::Parameters_Solver_BiCGStab_Cmplx() { append_entry(*this); }
//- end

//====================================================================
void Solver_BiCGStab_Cmplx::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  int    Niter;
  double Stop_cond;

  int err = 0;
  err += params.fetch_int("maximum_number_of_iteration", Niter);
  err += params.fetch_double("convergence_criterion_squared", Stop_cond);

  if (err) {
    vout.crucial(m_vl, "Solver_BiCGStab_Cmplx: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(Niter, Stop_cond);
}


//====================================================================
void Solver_BiCGStab_Cmplx::set_parameters(const int Niter, const double Stop_cond)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Solver_BiCGStab_Cmplx:\n");
  vout.general(m_vl, "  Niter     = %d\n", Niter);
  vout.general(m_vl, "  Stop_cond = %16.8e\n", Stop_cond);

  //- range check
  int err = 0;
  err += ParameterCheck::non_negative(Niter);
  err += ParameterCheck::square_non_zero(Stop_cond);

  if (err) {
    vout.crucial(m_vl, "Solver_BiCGStab_Cmplx: parameter range check failed.\n");
    abort();
  }

  //- store values
  m_Niter     = Niter;
  m_Stop_cond = Stop_cond;
}


//====================================================================
void Solver_BiCGStab_Cmplx::solve(Field& xq, const Field& b,
                                  int& Nconv, double& diff)
{
  vout.paranoiac(m_vl, "  BiCGStab_Cmplx solver starts\n");

  reset_field(b);

  vout.paranoiac(m_vl, "    norm of b = %16.8e\n", b.norm2());
  vout.paranoiac(m_vl, "    size of b = %d\n", b.size());

  reset_field(b);

  double snorm = 1.0 / b.norm2();
  double rr;

  Nconv = -1;
  s     = b;

  solve_init(b, rr);

  vout.detailed(m_vl, "    iter: %8d  %22.15e\n", 0, rr * snorm);

  for (int iter = 0; iter < m_Niter; iter++) {
    solve_step(rr);

    vout.detailed(m_vl, "    iter: %8d  %22.15e\n", 2 * (iter + 1), rr * snorm);

    if (rr * snorm < m_Stop_cond) {
      s    = m_fopr->mult(x);
      s   -= b;
      diff = s.norm();

      if (diff * diff * snorm < m_Stop_cond) {
        Nconv = 2 * (iter + 1);
        break;
      }

      s = x;
      solve_init(b, rr);
    }
  }
  if (Nconv == -1) {
    vout.crucial(m_vl, "BiCGStab_Cmplx solver not converged.\n");
    abort();
  }

  p    = m_fopr->mult(x);
  p   -= b;
  diff = p.norm();

  xq = x;
}


//====================================================================
void Solver_BiCGStab_Cmplx::reset_field(const Field& b)
{
  int Nin  = b.nin();
  int Nvol = b.nvol();
  int Nex  = b.nex();

  if ((s.nin() != Nin) || (s.nvol() != Nvol) || (s.nex() != Nex)) {
    s.reset(Nin, Nvol, Nex);
    r.reset(Nin, Nvol, Nex);
    x.reset(Nin, Nvol, Nex);
    p.reset(Nin, Nvol, Nex);
    v.reset(Nin, Nvol, Nex);

    w.reset(Nin, Nvol, Nex);

    rh.reset(Nin, Nvol, Nex);

    vout.paranoiac(m_vl, "    Solver_BiCGStab_Cmplx: field size reset.\n");
  }
}


//====================================================================
void Solver_BiCGStab_Cmplx::solve_init(const Field& b, double& rr)
{
  v  = m_fopr->mult(s);
  r  = b;
  x  = s;
  r -= v;
  rh = r;
  rr = r * r;

  rho_p   = cmplx(1.0, 0.0);
  alpha_p = cmplx(1.0, 0.0);
  omega_p = cmplx(1.0, 0.0);

  p = 0.0;
  v = 0.0;

  w = 0.0;
}


//====================================================================
void Solver_BiCGStab_Cmplx::solve_step(double& rr)
{
  // dcomplex rho = rh * r;
  double const_r, const_i;

  innerprod_c(const_r, const_i, rh, r);
  dcomplex rho = cmplx(const_r, const_i);

  dcomplex bet = rho * alpha_p / (rho_p * omega_p);

  // p = r + bet * (p - omega_p * v);
  mult_c(w, v, real(omega_p), imag(omega_p));
  v = p - w;
  mult_c(w, v, real(bet), imag(bet));
  p = r + w;

  v = m_fopr->mult(p);

  // dcomplex aden = rh * v;
  innerprod_c(const_r, const_i, rh, v);
  dcomplex aden = cmplx(const_r, const_i);

  dcomplex alpha = rho / aden;

  // s = r - ( alpha * v);
  mult_c(w, v, real(alpha), imag(alpha));
  s = r - w;

  Field t(s);
  t = m_fopr->mult(s);

  double omega_d = t * t;

  // dcomplex omega_n = t * s;
  innerprod_c(const_r, const_i, t, s);
  dcomplex omega_n = cmplx(const_r, const_i);

  dcomplex omega = omega_n / omega_d;

  // x += omega * s + alpha * p;
  mult_c(w, s, real(omega), imag(omega));
  x += w;
  mult_c(w, p, real(alpha), imag(alpha));
  x += w;

  // r = s - omega * t;
  mult_c(w, t, real(omega), imag(omega));
  r = s - w;

  rho_p   = rho;
  alpha_p = alpha;
  omega_p = omega;

  rr = r * r;
}


//====================================================================
void Solver_BiCGStab_Cmplx::innerprod_c(double& prod_r, double& prod_i,
                                        const Field& v, const Field& w)
{
  // prod = (v,w);

  int size = w.size();

  assert(v.size() == size);

  prod_r = 0.0;
  prod_i = 0.0;

  for (int i = 0; i < size; i += 2) {
    prod_r += v.cmp(i) * w.cmp(i) + v.cmp(i + 1) * w.cmp(i + 1);
    prod_i += v.cmp(i) * w.cmp(i + 1) - v.cmp(i + 1) * w.cmp(i);
  }

  prod_r = Communicator::reduce_sum(prod_r);
  prod_i = Communicator::reduce_sum(prod_i);
}


//====================================================================
void Solver_BiCGStab_Cmplx::mult_c(Field& v,
                                   const Field& w,
                                   const double& prod_r, const double& prod_i)
{
  // v = dcomplex(prod_r,prod_i) * w;

  int size = w.size();

  assert(v.size() == size);

  double v_r, v_i;
  for (int i = 0; i < size; i += 2) {
    v_r = prod_r * w.cmp(i) - prod_i * w.cmp(i + 1);
    v_i = prod_r * w.cmp(i + 1) + prod_i * w.cmp(i);

    v.set(i, v_r);
    v.set(i + 1, v_i);
  }
}


//====================================================================
//============================================================END=====
