/*!
        @file    $Id:: solver_CG.cpp #$

        @brief

        @author  <Hideo Matsufuru> hideo.matsufuru@kek.jp(matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate:: 2013-07-19 14:15:23 #$

        @version $LastChangedRevision: 936 $
*/

#include "solver_CG.h"

#ifdef USE_PARAMETERS_FACTORY
#include "parameters_factory.h"
#endif

#ifdef USE_FACTORY
namespace {
  Solver *create_object(Fopr *fopr)
  {
    return new Solver_CG(fopr);
  }


  bool init = Solver::Factory::Register("CG", create_object);
}
#endif

//- parameter entries
namespace {
  void append_entry(Parameters& param)
  {
    param.Register_int("maximum_number_of_iteration", 0);
    param.Register_double("convergence_criterion_squared", 0.0);

    param.Register_string("verbose_level", "NULL");
  }


#ifdef USE_PARAMETERS_FACTORY
  bool init_param = ParametersFactory::Register("Solver.CG", append_entry);
#endif
}
//- end

//- parameters class
Parameters_Solver_CG::Parameters_Solver_CG() { append_entry(*this); }
//- end

//====================================================================
void Solver_CG::set_parameters(const Parameters& params)
{
  const string str_vlevel = params.get_string("verbose_level");

  m_vl = vout.set_verbose_level(str_vlevel);

  //- fetch and check input parameters
  int    Niter;
  double Stop_cond;

  int err = 0;
  err += params.fetch_int("maximum_number_of_iteration", Niter);
  err += params.fetch_double("convergence_criterion_squared", Stop_cond);

  if (err) {
    vout.crucial(m_vl, "Solver_CG: fetch error, input parameter not found.\n");
    abort();
  }


  set_parameters(Niter, Stop_cond);
}


//====================================================================
void Solver_CG::set_parameters(const int Niter, const double Stop_cond)
{
  //- print input parameters
  vout.general(m_vl, "Parameters of Solver_CG:\n");
  vout.general(m_vl, "  Niter     = %d\n", Niter);
  vout.general(m_vl, "  Stop_cond = %16.8e\n", Stop_cond);

  //- range check
  int err = 0;
  err += ParameterCheck::non_negative(Niter);
  err += ParameterCheck::square_non_zero(Stop_cond);

  if (err) {
    vout.crucial(m_vl, "Solver_CG: parameter range check failed.\n");
    abort();
  }

  //- store values
  m_Niter     = Niter;
  m_Stop_cond = Stop_cond;
}


//====================================================================
void Solver_CG::solve(Field& xq, const Field& b,
                      int& Nconv, double& diff)
{
  vout.detailed(m_vl, "  CG solver starts\n");

  reset_field(b);

  vout.paranoiac(m_vl, "    norm of b = %16.8e\n", b.norm2());
  vout.paranoiac(m_vl, "    size of b = %d\n", b.size());

  double snorm = 1.0 / b.norm2();
  double rr;

  Nconv = -1;
  s     = b;

  solve_init(b, rr);

  vout.detailed(m_vl, "    iter: %8d  %22.15e\n", 0, rr * snorm);

  for (int iter = 0; iter < m_Niter; iter++) {
    solve_step(rr);

    vout.detailed(m_vl, "    iter: %8d  %22.15e\n", (iter + 1), rr * snorm);

    if (rr * snorm < m_Stop_cond) {
      // s = m_fopr->mult(x);
      m_fopr->mult(s, x);
      s   -= b;
      diff = s.norm();

      if (diff * diff * snorm < m_Stop_cond) {
        Nconv = iter + 1;
        break;
      }

      s = x;
      solve_init(b, rr);
    }
  }
  if (Nconv == -1) {
    vout.crucial(m_vl, "CG solver not converged.\n");
    abort();
  }

  // p = m_fopr->mult(x);
  m_fopr->mult(p, x);
  p   -= b;
  diff = p.norm();

  xq = x;
}


//====================================================================
void Solver_CG::reset_field(const Field& b)
{
  int Nin  = b.nin();
  int Nvol = b.nvol();
  int Nex  = b.nex();

  if ((s.nin() != Nin) || (s.nvol() != Nvol) || (s.nex() != Nex)) {
    s.reset(Nin, Nvol, Nex);
    r.reset(Nin, Nvol, Nex);
    x.reset(Nin, Nvol, Nex);
    p.reset(Nin, Nvol, Nex);
    v.reset(Nin, Nvol, Nex);

    vout.paranoiac(m_vl, "  Solver_CG: field size reset.\n");
  }
}


//====================================================================
void Solver_CG::solve_init(const Field& b, double& rr)
{
  r = b;
  x = s;

  //  s = m_fopr->mult(x);
  m_fopr->mult(s, x);

  r -= s;
  p  = r;
  rr = r * r;
}


//====================================================================
void Solver_CG::solve_step(double& rr)
{
  // s = m_fopr->mult(p);
  m_fopr->mult(s, p);

  double pap  = p * s;
  double rr_p = rr;
  double cr   = rr_p / pap;

  v  = p;
  v *= cr;
  x += v;
  //  x += cr*p;

  s *= cr;
  r -= s;

  rr = r * r;
  p *= rr / rr_p;
  p += r;
}


//====================================================================
//============================================================END=====
