/*!
        @file    $Id: test_Eigensolver.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: aoym $

        @date    $LastChangedDate: 2013-01-22 13:51:53 #$

        @version $LastChangedRevision: 875 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "eigensolver_IRLanczos.h"

#include "gaugeConfig.h"
#include "vec_SU_N.h"

#include "fopr_Clover.h"

#include "projection.h"
#include "smear.h"
#include "sortField.h"

#include "director_Smear.h"
#include "fopr_Smeared.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of eigenvalue solver.

/*!
                                          [28 Dec 2011 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.                  [14 Nov 2012 Y.Namekawa]
    Selectors are implemented.            [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

namespace Test_Eigensolver {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_Eigensolver.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_Eigensolver : public Parameters {
     public:
      Parameters_Test_Eigensolver()
      {
        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int solve(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "Eigensolver.Solve",
      solve
      );
  }
#endif

  //====================================================================
  //- Check of eigenvalue solver
  int solve(void)
  {
    // #####  parameter setup  #####
    int Ndim = CommonParameters::Ndim();
    int Nvol = CommonParameters::Nvol();

    Parameters *params_test      = new Parameters_Test_Eigensolver;
    Parameters *params_clover    = ParametersFactory::New("Fopr.Clover");
    Parameters *params_proj      = ParametersFactory::New("Projection");
    Parameters *params_smear     = ParametersFactory::New("Smear");
    Parameters *params_dr_smear  = ParametersFactory::New("Director_Smear");
    Parameters *params_irlanczos = ParametersFactory::New("Eigensolver");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_Eigensolver", params_test);
    params_all->Register_Parameters("Fopr_Clover", params_clover);
    params_all->Register_Parameters("Projection", params_proj);
    params_all->Register_Parameters("Smear", params_smear);
    params_all->Register_Parameters("Director_Smear", params_dr_smear);
    params_all->Register_Parameters("Eigensolver", params_irlanczos);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_read = params_test->get_string("gauge_config_type_input");
    const string readfile       = params_test->get_string("config_filename_input");
    const string str_vlevel     = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type     = params_clover->get_string("gamma_matrix_type");
    const string str_proj_type      = params_proj->get_string("projection_type");
    const string str_smear_type     = params_smear->get_string("smear_type");
    const string str_sortfield_type = params_irlanczos->get_string("eigensolver_mode");
    const int    Nk = params_irlanczos->get_int("number_of_wanted_eigenvectors");
    const int    Np = params_irlanczos->get_int("number_of_working_eigenvectors");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_read     = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile       = %s\n", readfile.c_str());
    vout.general(vl, "  vlevel         = %s\n", str_vlevel.c_str());
    vout.general(vl, "  gmset_type     = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  proj_type      = %s\n", str_proj_type.c_str());
    vout.general(vl, "  smear_type     = %s\n", str_smear_type.c_str());
    vout.general(vl, "  sortfield_type = %s\n", str_sortfield_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);

    if (err) {
      vout.crucial(vl, "Test_Eigensolver: Input parameters have not been set.\n");
      abort();
    }


    // ####  Set up a gauge configuration  ####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);
    gconf_read->read_file((Field *)U, readfile);
    // gconf_read->set_cold((Field*)U);

    Projection *proj  = Projection::New(str_proj_type);
    Smear      *smear = Smear::New(str_smear_type, proj);
    smear->set_parameters(*params_smear);

    Director_Smear *dr_smear = new Director_Smear((Smear *)smear);
    dr_smear->set_parameters(*params_dr_smear);


    // ####  object setup  #####
    Fopr_Clover *fopr_c = new Fopr_Clover(str_gmset_type);
    fopr_c->set_parameters(*params_clover);

    Fopr_Smeared *fopr_smear = new Fopr_Smeared(fopr_c, dr_smear);
    Fopr         *fopr       = fopr_smear;
    fopr->set_config(U);
    fopr->set_mode("H");

    SortField *sort = SortField::New(str_sortfield_type);

    Eigensolver_IRLanczos *eigen = new Eigensolver_IRLanczos(fopr, sort);
    eigen->set_parameters(*params_irlanczos);


    // ####  Execution main part  ####
    // int Nm = 100;
    int              Nm = Nk + Np;
    valarray<double> TDa(Nm);
    valarray<Field>  vk(Nm);

    Field_F b2;
    int     NFin  = b2.nin();
    int     NFvol = b2.nvol();
    int     NFex  = b2.nex();
    for (int k = 0; k < Nm; ++k) {
      vk[k].reset(NFin, NFvol, NFex);
    }

    int Nsbt  = -1;
    int Nconv = -100;
    eigen->solve(TDa, vk, Nsbt, Nconv, (Field)b2);

    Field v;
    v.reset(NFin, NFvol, NFex);
    double vv = 0.0;  // superficial initialization

    for (int i = 0; i < Nsbt + 1; ++i) {
      v  = fopr->mult(vk[i]);
      v -= TDa[i] * vk[i];
      vv = v * v;

      vout.general(vl, "Eigenvalues: %4d %20.14f %20.15e \n", i, TDa[i], vv);
    }

    double result = vv;


    // #####  tidy up  #####
    delete U;
    delete gconf_read;

    delete dr_smear;

    delete fopr_c;

    delete proj;
    delete smear;
    delete sort;

    delete eigen;

    delete params_test;
    delete params_clover;
    delete params_proj;
    delete params_smear;
    delete params_dr_smear;
    delete params_irlanczos;
    delete params_all;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_Eigensolver
