/*!
        @file    $Id: test_HMC_Clover_Leapfrog_Nf2.cpp #$

        @brief

        @author  Hideo Matsufuru <hideo.matsufuru@kek.jp> (matsufuru)
                 $LastChangedBy: sueda $

        @date    $LastChangedDate: 2013-01-22 13:51:53 #$

        @version $LastChangedRevision: 936 $
*/

#include "parameterManager_YAML.h"
#include "parameters_factory.h"

#include "bridgeIO.h"
using Bridge::vout;

#include "gaugeConfig.h"

#include "action_G_Plaq.h"
#include "action_G_Rectangle.h"

#include "fopr_Clover.h"
#include "force_F_Clover_Nf2.h"

#include "randomNumbers_Mseries.h"

#include "director_Smear.h"
#include "fopr_Smeared.h"
#include "force_F_Smeared.h"

#include "action_F_Standard_lex.h"
#include "fprop_Standard_lex.h"

#include "forceSmear.h"
#include "projection.h"
#include "smear.h"
#include "solver.h"

#include "hmc_Leapfrog.h"

#ifdef USE_TEST
#include "test.h"
#endif

#ifdef USE_TESTMANAGER_AUTOREGISTER
#include "testManager.h"
#endif

//====================================================================
//! Test of HMC update for clover fermions.

/*!
    This class tests HMC update for dynamical clover fermions.
    Smearing of gauge configuration for the fermion operator
    is incorporated.
                                         [12 Apr 2012 H.Matsufuru]
    (Coding history will be recovered from trac.)
    YAML is implemented.                 [14 Nov 2012 Y.Namekawa]
    Fprop and selectors are implemented. [03 Mar 2013 Y.Namekawa]
    (Selectors are replaced with factories by Aoyama-san)
 */

namespace Test_HMC_Clover {
  //- test-private parameters
  namespace {
    const std::string filename_input  = "test_HMC_Clover_Leapfrog_Nf2.yaml";
    const std::string filename_output = "stdout";

    class Parameters_Test_HMC_Clover : public Parameters {
     public:
      Parameters_Test_HMC_Clover()
      {
        Register_string("gauge_config_status", "NULL");

        Register_string("gauge_config_type_input", "NULL");
        Register_string("config_filename_input", "NULL");

        Register_string("gauge_config_type_output", "NULL");
        Register_string("config_filename_output", "NULL");

        Register_int("trajectory_number", 0);
        Register_int("trajectory_number_step", 0);
        Register_int("save_config_interval", 0);

        Register_string("verbose_level", "NULL");

        Register_double("expected_result", 0.0);
      }
    };
  }

  //- prototype declaration
  int leapfrog_Nf2(void);

#ifdef USE_TESTMANAGER_AUTOREGISTER
  namespace {
    static const bool is_registered = TestManager::RegisterTest(
      "HMC.Clover.Leapfrog_Nf2",
      leapfrog_Nf2
      );
  }
#endif

  //====================================================================
  int leapfrog_Nf2(void)
  {
    // #####  parameter setup  #####
    int Nc   = CommonParameters::Nc();
    int Nvol = CommonParameters::Nvol();
    int Ndim = CommonParameters::Ndim();
    int NinG = 2 * Nc * Nc;

    Parameters *params_test      = new Parameters_Test_HMC_Clover;
    Parameters *params_action_G  = ParametersFactory::New("Action.G_Rectangle");
    Parameters *params_clover    = ParametersFactory::New("Fopr.Clover");
    Parameters *params_proj      = ParametersFactory::New("Projection");
    Parameters *params_smear     = ParametersFactory::New("Smear");
    Parameters *params_dr_smear  = ParametersFactory::New("Director_Smear");
    Parameters *params_solver_MD = ParametersFactory::New("Solver");
    Parameters *params_solver_H  = ParametersFactory::New("Solver");
    Parameters *params_hmc       = ParametersFactory::New("HMC.Leapfrog");

    Parameters *params_all = new Parameters;

    params_all->Register_Parameters("Test_HMC_Clover", params_test);
    params_all->Register_Parameters("Action_G_Rectangle", params_action_G);
    params_all->Register_Parameters("Fopr_Clover", params_clover);
    params_all->Register_Parameters("Projection", params_proj);
    params_all->Register_Parameters("Smear", params_smear);
    params_all->Register_Parameters("Director_Smear", params_dr_smear);
    params_all->Register_Parameters("Solver_MD", params_solver_MD);
    params_all->Register_Parameters("Solver_H", params_solver_H);
    params_all->Register_Parameters("HMC_Leapfrog", params_hmc);

    ParameterManager_YAML params_manager;
    params_manager.read_params(filename_input, params_all);

    const string str_gconf_status = params_test->get_string("gauge_config_status");
    const string str_gconf_read   = params_test->get_string("gauge_config_type_input");
    const string readfile         = params_test->get_string("config_filename_input");
    const string str_gconf_write  = params_test->get_string("gauge_config_type_output");
    const string writefile        = params_test->get_string("config_filename_output");
    int          iconf            = params_test->get_int("trajectory_number");
    const int    Ntraj            = params_test->get_int("trajectory_number_step");
    const int    i_save_conf      = params_test->get_int("save_config_interval");
    const string str_vlevel       = params_test->get_string("verbose_level");
#ifdef USE_TEST
    const double expected_result = params_test->get_double("expected_result");
#endif

    const string str_gmset_type     = params_clover->get_string("gamma_matrix_type");
    const string str_proj_type      = params_proj->get_string("projection_type");
    const string str_smear_type     = params_smear->get_string("smear_type");
    const string str_solver_MD_type = params_solver_MD->get_string("solver_type");
    const string str_solver_H_type  = params_solver_H->get_string("solver_type");

    Bridge::VerboseLevel vl = vout.set_verbose_level(str_vlevel);

    //- print input parameters
    vout.general(vl, "  gconf_status = %s\n", str_gconf_status.c_str());
    vout.general(vl, "  gconf_read     = %s\n", str_gconf_read.c_str());
    vout.general(vl, "  readfile       = %s\n", readfile.c_str());
    vout.general(vl, "  gconf_write    = %s\n", str_gconf_write.c_str());
    vout.general(vl, "  writefile      = %s\n", writefile.c_str());
    vout.general(vl, "  iconf          = %d\n", iconf);
    vout.general(vl, "  Ntraj          = %d\n", Ntraj);
    vout.general(vl, "  i_save_conf    = %d\n", i_save_conf);
    vout.general(vl, "  vlevel         = %s\n", str_vlevel.c_str());

    vout.general(vl, "  gmset_type     = %s\n", str_gmset_type.c_str());
    vout.general(vl, "  proj_type      = %s\n", str_proj_type.c_str());
    vout.general(vl, "  smear_type     = %s\n", str_smear_type.c_str());
    vout.general(vl, "  solver_MD_type = %s\n", str_solver_MD_type.c_str());
    vout.general(vl, "  solver_H_type  = %s\n", str_solver_H_type.c_str());
    vout.general(vl, "\n");

    //- input parameter check
    int err = 0;
    err += ParameterCheck::non_NULL(str_gconf_read);
    err += ParameterCheck::non_NULL(readfile);
    err += ParameterCheck::non_NULL(str_gconf_write);
    err += ParameterCheck::non_NULL(writefile);
    err += ParameterCheck::non_zero(iconf);
    err += ParameterCheck::non_zero(Ntraj);
    err += ParameterCheck::non_zero(i_save_conf);

    if (err) {
      vout.crucial(vl, "test_HMC_Clover: Input parameters have not been set.\n");
      abort();
    }


    // #####  object setup  #####
    Field_G     *U          = new Field_G(Nvol, Ndim);
    GaugeConfig *gconf_read = new GaugeConfig(str_gconf_read);

    if (str_gconf_status == "Continue") {
      gconf_read->read_file((Field *)U, readfile);
    } else if (str_gconf_status == "Start_cold") {
      gconf_read->set_cold((Field *)U);
    } else {
      vout.crucial(vl, "Test_HMC_Clover: unsupported gconf status \"%s\".\n", str_gconf_status.c_str());
      abort();
    }


    Action_G_Rectangle *action_G = new Action_G_Rectangle; // plaq.+rectangle (SA)
    action_G->set_parameters(*params_action_G);


    Fopr_Clover *fopr_c = new Fopr_Clover(str_gmset_type); // define fermion operator (SA)
    fopr_c->set_parameters(*params_clover);

    Force_F_Clover_Nf2 *force_fopr_c = new Force_F_Clover_Nf2(str_gmset_type);
    // define fermion force (SA)
    force_fopr_c->set_parameters(*params_clover);

    // define smearing method (SA)
    Projection *proj  = Projection::New(str_proj_type);
    Smear      *smear = Smear::New(str_smear_type, proj);
    smear->set_parameters(*params_smear);

    // define force smearing method (SA)
    ForceSmear *force_smear = ForceSmear::New(str_smear_type, proj);
    force_smear->set_parameters(*params_smear);

    Director_Smear *dr_smear = new Director_Smear((Smear *)smear);
    dr_smear->set_parameters(*params_dr_smear);

    Fopr_Smeared *fopr_smear = new Fopr_Smeared((Fopr *)fopr_c, dr_smear);
    // define smeared fermion operator (SA)
    Force_F_Smeared *force_fopr_smear
      = new Force_F_Smeared((Force *)force_fopr_c, (ForceSmear *)force_smear, dr_smear);
    // define smeared fermion force (SA)

    Solver *solver_MD = Solver::New(str_solver_MD_type, fopr_smear);
    solver_MD->set_parameters(*params_solver_MD);
    Fprop *fprop_MD = new Fprop_Standard_lex(solver_MD);

    Solver *solver_H = Solver::New(str_solver_H_type, fopr_smear);
    solver_H->set_parameters(*params_solver_H);
    Fprop *fprop_H = new Fprop_Standard_lex(solver_H);

    Action_F_Standard_lex *action_F
      = new Action_F_Standard_lex((Fopr *)fopr_smear, (Force *)force_fopr_smear,
                                  fprop_MD, fprop_H);
    // define fermion action (SA)


    valarray<Action *> actions(2);
    actions[0] = (Action *)action_G; // register action[0] (SA)
    actions[1] = (Action *)action_F; // register action[1] (SA)

    valarray<Director *> directors(1);
    directors[0] = (Director *)dr_smear; // register director[0] (SA)


    //- Random number is initialized with a parameter specified by iconf
    RandomNumbers *rand = new RandomNumbers_Mseries(iconf);


    HMC_Leapfrog hmc(actions, directors, rand); // define hmc_leapfrog (SA)
    hmc.set_parameters(*params_hmc);


    // ####  Execution main part  ####
    vout.general(vl, "HMC: Ntraj = %d\n", Ntraj); // a number of trajectory (SA)

    double result = 0.0;
    for (int traj = 0; traj < Ntraj; ++traj) {
      vout.general(vl, "\n");
      vout.general(vl, "traj = %d\n", traj);

      result = hmc.update(*U); // hmc update (SA)
    }


    // #####  tidy up  #####
    delete U;
    delete gconf_read;

    delete rand;

    delete action_G;
    delete action_F;

    delete       fopr_c;
    delete force_fopr_c;

    delete proj;
    delete            smear;
    delete      force_smear;
    delete         dr_smear;
    delete       fopr_smear;
    delete force_fopr_smear;

    delete solver_MD;
    delete  fprop_MD;
    delete solver_H;
    delete  fprop_H;

    delete params_test;
    delete params_action_G;
    delete params_clover;
    delete params_proj;
    delete params_smear;
    delete params_dr_smear;
    delete params_solver_MD;
    delete params_solver_H;
    delete params_hmc;
    delete params_all;


#ifdef USE_TEST
    return Test::verify(expected_result, result);

#else
    return EXIT_SUCCESS;
#endif
  }
} // namespace Test_HMC_Clover
